;################################################################################
;#										#
;# tile mode 7 engine								#
;# copyright (c) 2005-2009 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# This library is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU Lesser General Public			#
;# License as published by the Free Software Foundation; either			#
;# version 3 of the License, or (at your option) any later version.		#
;#										#
;# This library is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU		#
;# Lesser General Public License for more details.				#
;#										#
;# You should have received a copy of the GNU Lesser General Public		#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

.include	"/usr/local/include/avr/m644Pdef.inc"
.include	"../../../system/includes/api_macros.inc"
.include	"../../../system/includes/libdef.asm"
.include	"../../../system/includes/definitions.asm"

.equ		tm_y_start	= libmio_vram	;y start
.equ		tm_x_start	= libmio_vram+1	;x start
.equ		temp_tile	= libmio_vram+2	;temp tile

.CSEG

.org	0x7200
;                   "......      N"  
start:		.db " Tile  -01- N",0xf3,0xfe,0x90	;video driver
;icon def
		.db 0xf2,0xf1,0xf1,0xf3
		.db 0xf0,"VM",0xf0
		.db 0xf4,0xf1,0xf1,0xf5

.org	0x720e
		rjmp	inex			;IN 5xx
		rjmp	outex			;OUT 5xx
		ret				;FKT 00 ID
		rjmp	tilemode_init		;FKT 01 init
		rjmp	tilemode_cls		;FKT 02 clear screen
		rjmp	tilemode_char		;FKT 03 output char
		rjmp	tilemode_gxy		;FKT 04 set cursor
		ret				;FKT 05 plot
		ret				;FKT 06 new line
		rjmp	tilemode_vid		;FKT 07 video display
		ret				;FKT 08 startup

;specific functions
		rjmp	tilemode_fmap		;FKT 09 fill map (value)
		ret				;FKT 10 fill map from array (one tile) 
		ret				;FKT 11 fill map area
		rjmp	tilemode_smap		;FKT 12 set map (Y,X,value)
		rjmp	tilemode_gmap		;FKT 13 get map (Y,X) to ~R
		rjmp	tile_solid		;FKT 14 fill tile N solid with COLOR c
		rjmp	tile_fpattern		;FKT 15 draw tile N with flash pattern M COLOR C
		rjmp	tile_rpattern		;FKT 16 draw tile N with ram pattern from array cell M COLOR C

		rjmp	tile_cpattern		;FKT 17 draw tile N with char cell M COLOR C
		rjmp	tile_xflip		;FKT 18 flip tile horizontal
		rjmp	tile_yflip		;FKT 19 flip tile vertical
		rjmp	tile_cw			;FKT 20 rotate tile clockwise
		rjmp	tile_ccw		;FKT 21 rotate tile counterclockwise
		rjmp	tile_copy		;FKT 22 tile copy (T1,T2)
		rjmp	tile_add		;FKt 23 add tile information (T1,T2,Tcolor)
		rjmp	tile_rot		;FKT 24 rotate 2 tiles (T1,T2,DIR)

		rjmp	tile_spinit		;FKT 25 init sprite area
		rjmp	tile_stype		;FKT 26 set sprite tile (sprite,tile,transcolor) and activate
		rjmp	tile_scoords		;FKT 27 set coordinates and direction
		rjmp	tile_senable		;FKT 28 activate sprite in sequence
		rjmp	tile_sdisable		;FKT 29 de-actiate sprite in sequence
		rjmp	ti_spdraw		;FKT 30 draw sprite
		rjmp	ti_undraw		;FKT 31 undraw sprite
		rjmp	tile_dseq		;FKT 32 draw sprite sequence
		rjmp	tile_useq		;FKT 33 undraw sprite sequence
		rjmp	tile_movea		;FKT 34 move all sprites (set coords)

		rjmp	tile_movea_drw		;FKT 35 move all sprites and draw
		ret				;FKT 36
		ret				;FKT 37

inex:		ldi	ereg,40			;no IO driver
		ret

outex:		ldi	ereg,40			;no IO driver
		ret

retzero:	clr	XL
		clr	XH
retvalue:	api_getvalues
		st	Y,XL
		std	Y+1,XH
		ret

;upper text line
tilemode_t1:	ldi	XL,LOW(libmio_vram+2632)	;1
		ldi	XH,HIGH(libmio_vram+2632)	;1
		ldi	YL,LOW(libmio_vram+2692)	;1
		ldi	YH,HIGH(libmio_vram+2692)	;1
		subi	tempreg1,0xf6			;1 +10
		sts	libmio_cline,tempreg1		;2
		api_vm0line				;view textline

;lower text line
tilemode_t2:	ldi	XL,LOW(libmio_vram+2662)	;1
		ldi	XH,HIGH(libmio_vram+2662)	;1
		ldi	YL,LOW(libmio_vram+2722)	;1
		ldi	YH,HIGH(libmio_vram+2722)	;1
		subi	tempreg1,208			;1
		sts	libmio_cline,tempreg1		;2
		api_vm0line				;view textline

tilemode_novid:	ret

tilemode_vid:	mov	tempreg1,vline_l	;y-line
		subi	tempreg1,5
		brcs	tilemode_novid
		subi	tempreg1,10		;upper text?
		brcs	tilemode_t1		;yes
		cpi	tempreg1,208		;lower text?
		brcc	tilemode_t2		;yes
		push	tempreg4		;save additional registers
		push	tempreg5
		push	tempreg6
		push	tempreg7

		lds	tempreg7,libmio_border	;get border color

		lsr	tempreg1		;line / 2 (0...206)
		lds	XH,tm_y_start		;Y start
		add	XH,tempreg1		;this is the line to display
		mov	tempreg2,XH		;copy
		andi	tempreg2,0x07		;this is the pixline
		lsl	tempreg2		;4 bytes per pixline
		lsl	tempreg2		;
		mov	tempreg6,tempreg2	;copy tile offset

		andi	XH,0xf8			;this is the tile line *8
		ldi	XL,4			;32 tiles hor (/8)
		mul	XL,XH			;offset
		mov	XH,r1
		subi	XH,0xfe			;start at 0x200
		mov	XL,r0
		mov	tempreg5,XL		;save this as start mask

		lds	YL,tm_x_start		;X start
		mov	tempreg2,YL		;copy
		lsr	YL			;tempreg1=tile offset
		lsr	YL
		lsr	YL
		add	XL,YL			;set new address

;tempreg7 = border color
;tempreg6 = tiledata-offset
;tempreg5 = HIGH-Bits from LOW-byte of tilemap

		ldi	tempreg4,32		;1 bytes/Tile
		ld	tempreg3,X		;2 get first data
		andi	tempreg3,0x3f		;1 limit data
		mul	tempreg3,tempreg4	;2 calc tiletable offset
		movw	YL,r0			;2 offset
		add	YL,tempreg6		;1 add pixline offset
		subi	YH,0xfa			;1 add base 0x600
		sbrc	tempreg2,1
		inc	YL
		sbrc	tempreg2,2
		subi	YL,0xfe
		ld	tempreg1,Y+		;2 get pixel byte P0/P1 of next

		andi	tempreg2,0x07
		brne	tilemode_v1
		dec	XL

tilemode_v1:	inc	XL			;1 next tile
		andi	XL,0x1f			;1 limit to 32 tiles
		or	XL,tempreg5		;1 add offset
		ld	tempreg3,X		;2
		lsl	tempreg2
		lsl	tempreg2
		lsl	tempreg2
		lds	ZH,prg_vdrv
		ldi	ZL,LOW(ti_jtab)
		add	ZL,tempreg2
		adc	ZH,const_0
		ldi	tempreg2,127		;1 16*8-1 (visible size)
		ijmp

ti_jtab:
;pix 0
		ld	tempreg3,X		;2 (filling)
		ld	tempreg3,X		;2 (filling)
		nop				;1 filling
		nop				;1 filling
		nop				;1 filling
		rjmp	bloop_0			;2 jump to pixel output
		rjmp	bloop_0			;(dummy)
		nop				;(dummy)

;pix 1
		ld	tempreg3,X		;2 (filling)
		ld	tempreg3,X		;2 (filling)
		nop				;1 filling
		nop				;1 filling
		swap	tempreg1		;1
		rjmp	bloop_1			;2 jump to pixel output
		rjmp	bloop_1			;(dummy)
		nop				;(dummy)

;pix 2
		ld	tempreg3,X		;2 (filling)
		ld	tempreg3,X		;2 (filling)
		nop				;1 filling
		nop				;1 filling
		nop				;1 filling
		rjmp	bloop_2			;2 jump to pixel output
		rjmp	bloop_2			;(dummy)
		nop				;(dummy)

;pix 3
		ld	tempreg3,X		;2 (filling)
		nop				;1 filling
		nop				;1 filling
		ld	tempreg3,X		;2 get next data
		swap	tempreg1		;1 prepare P1
		rjmp	bloop_3			;2 jump to pixel output
		rjmp	bloop_3			;(dummy)
		nop				;(dummy)

;pix4
		ld	tempreg3,X		;2 (filling)
		nop				;1 filling
		nop				;1 filling
		ld	tempreg3,X		;2 get next data
		andi	tempreg3,0x3f		;1 limit data
		rjmp	bloop_4			;2 jump to pixel output
		rjmp	bloop_4			;(dummy)
		nop				;(dummy)

;pix5
		nop				;1 filling
		ld	tempreg3,X		;2 get next data
		andi	tempreg3,0x3f		;1 limit data
		mul	tempreg3,tempreg4	;2 calc tiletable offset
		swap	tempreg1		;1 prepare P1
		rjmp	bloop_5			;2 jump to pixel output
		rjmp	bloop_5			;(dummy)
		nop				;(dummy)

;pix 6
		nop				;1 filling
		ld	tempreg3,X		;2 get next data
		andi	tempreg3,0x3f		;1 limit data
		mul	tempreg3,tempreg4	;2 calc tiletable offset
		movw	YL,r0			;2 offset
		rjmp	bloop_6			;2 jump to pixel output
		nop				;(dummy)
		nop				;(dummy)

;pix 7
		andi	tempreg3,0x3f		;1 limit data
		mul	tempreg3,tempreg4	;2 calc tiletable offset
		movw	YL,r0			;2 offset
		add	YL,tempreg6		;1 add pixline offset
		subi	YH,0xfa			;1 add base 0x600
		swap	tempreg1		;1
		rjmp	bloop_7			;2 jump to pixel output
		nop				;(dummy)

;---- P0 ---------------------------------------------------------------
bloop_0:	out	PORTC,tempreg1		;1 out P0
		inc	XL			;1 next tile
		andi	XL,0x1f			;1 limit to 32 tiles
		swap	tempreg1		;1 prepare P1
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P1 ---------------------------------------------------------------
bloop_1:	out	PORTC,tempreg1		;1 out P1
		ld	tempreg1,Y+		;2 get pixel byte P2/P3
		or	XL,tempreg5		;1 add offset
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P2 ---------------------------------------------------------------
bloop_2:	out	PORTC,tempreg1		;1 out P2
		swap	tempreg1		;1 prepare P3
		ld	tempreg3,X		;2 get next data
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P3 ---------------------------------------------------------------
bloop_3:	out	PORTC,tempreg1		;1 out P3
		ld	tempreg1,Y+		;2 get pixel byte P4/P5
		andi	tempreg3,0x3f		;1 limit data
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P4 ---------------------------------------------------------------
bloop_4:	out	PORTC,tempreg1		;1 out P4
		swap	tempreg1		;1 prepare P5
		mul	tempreg3,tempreg4	;2 calc tiletable offset
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P5 ---------------------------------------------------------------
bloop_5:	out	PORTC,tempreg1		;1 out P5
		ld	tempreg1,Y+		;2 get pixel byte P6/P7
		movw	YL,r0			;2 offset
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P6 ---------------------------------------------------------------
bloop_6:	out	PORTC,tempreg1		;1 out P6
		swap	tempreg1		;1 prepare P7
		add	YL,tempreg6		;1 add pixline offset
		subi	YH,0xfa			;1 add base 0x600
		sub	tempreg2,const_1	;1 decrement pix to do
		brcs	bend			;1/2

;---- P7 ---------------------------------------------------------------
bloop_7:	out	PORTC,tempreg1		;1 out P7
		ld	tempreg1,Y+		;2 get pixel byte P0/P1 of next
		sub	tempreg2,const_1	;1 byte counter
		brcc	bloop_0			;2 byte loop
		nop				;1 filling
bend_1:		nop				;1 filling

;---- Border -----------------------------------------------------------
bend:		out	PORTC,tempreg7		;1 out border
		pop	tempreg7		;restore registers
		pop	tempreg6
		pop	tempreg5
		pop	tempreg4
		ret

;-------------------------------------------------------------------------------
; clear screen
;-------------------------------------------------------------------------------
tilemode_cls:	rcall	tilemode_ctxt		;clear text
		rcall	tilemode_cmap		;clear tile map
		rcall	tilemode_cts		;clear tile data
		sts	tm_y_start,const_0	;set viewpoint to 0,0
		sts	tm_x_start,const_0
		ret

;-------------------------------------------------------------------------------
; clear text area
;-------------------------------------------------------------------------------
tilemode_ctxt:	api_pushxyz
		ldi	YL,LOW(libmio_vram+2632)
		ldi	YH,HIGH(libmio_vram+2632)
		ldi	XL,60			;chars
		ldi	XH,32			;space
		lds	r0,libmio_color
tilemode_ctxt1:	std	Y+60,r0			;write attr
		st	Y+,XH			;write char
		dec	XL
		brne	tilemode_ctxt1
		api_popxyz

tilemode_cmap:	api_pushxyz
		ldi	YL,LOW(0x200)
		ldi	YH,HIGH(0x200)
		ldi	XL,32			;Y
tilemode_cmap1:	ldi	XH,32			;X
tilemode_cmap2:	st	Y+,const_0
		dec	XH
		brne	tilemode_cmap2
		dec	XL
		brne	tilemode_cmap1
		api_popxyz

tilemode_cts:	api_pushxyz
		ldi	YL,LOW(0x600)
		ldi	YH,HIGH(0x600)
		ldi	XL,49			;Tiles
tilemode_cts1:	ldi	XH,32			;X
tilemode_cts2:	st	Y+,const_0
		dec	XH
		brne	tilemode_cts2
		dec	XL
		brne	tilemode_cts1
		api_popxyz

;------------------------------------------------------------------------------
; set video mode
;------------------------------------------------------------------------------
tilemode_init:	brtc	tilemode_i_e		;no exit
		ldi	XL,libmio_v0cols
		sts	libmio_clipx2,XL
		ldi	XL,2
		sts	libmio_clipy2,XL
		rcall	tilemode_ctxt
		rcall	tilemode_cmap
		rcall	tilemode_cts
		ldi	XL,233
tilemode_i_e:	ret

;-------------------------------------------------------------------------------
; output character and calculate new position (tempreg1=char)
;-------------------------------------------------------------------------------
tilemode_char:	push	XL
		push	XH
		push	YL
		push	YH
		movw	XL,libmio_cur_x			;X
		ldi	YH,HIGH(libmio_vram+2632)	;home position
		ldi	YL,LOW(libmio_vram+2632)	;home position
		add	YL,XL				;position+x
		adc	YH,const_0
		ldi	XL,0
		sbrc	XH,0
		ldi	XL,30
		add	YL,XL			;position+(cols*y)
		adc	YH,const_0
		st	Y,tempreg1		;write char
		ldi	XH,60
		add	YL,XH			;position+(cols*y)
		adc	YH,const_0
		lds	XH,libmio_color
		st	Y,XH

		movw	XL,libmio_cur_x		;X
		inc	XL
		cpi	XL,30
		brcs	tilemode_chr_1
		clr	XL
		inc	XH
tilemode_chr_1:	andi	XH,1
		movw	libmio_cur_x,XL
		pop	YH
		pop	YL
		pop	XH
		pop	XL
		ret

;-------------------------------------------------------------------------------
; set cursor to position XL:XH
;-------------------------------------------------------------------------------
tilemode_gxy:	movw	libmio_cur_x,XL		;save X,Y-position
		ret

;-------------------------------------------------------------------------------
; fill map
;-------------------------------------------------------------------------------
tilemode_fmap:	api_getvalues
		ldd	tempreg2,Y+2
		cpi	tempreg2,0x31
		brcc	tile_smap_e
		ldi	YL,LOW(0x200)
		ldi	YH,HIGH(0x200)
		ldi	XL,32			;Y
tilemode_fmap1:	ldi	XH,32			;X
tilemode_fmap2:	st	Y+,tempreg2
		dec	XH
		brne	tilemode_fmap2
		dec	XL
		brne	tilemode_fmap1
		rjmp	retzero

;-------------------------------------------------------------------------------
; set map value
;-------------------------------------------------------------------------------
tilemode_smap:	api_getvalues
		ldd	tempreg2,Y+6		;tile no
		cpi	tempreg2,0x31
		brcc	tile_smap_e
		ldd	XH,Y+2			;Y
		andi	XH,0x1f
		ldi	XL,32
		mul	XL,XH
		ldd	XL,Y+4
		andi	XL,0x1f
		ldi	YL,LOW(0x200)
		ldi	YH,HIGH(0x200)
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0
		st	Y,tempreg2
		sts	temp_tile,tempreg2
		rjmp	retzero

tile_smap_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

;-------------------------------------------------------------------------------
; get map value
;-------------------------------------------------------------------------------
tilemode_gmap:	api_getvalues
		ldd	XH,Y+2			;Y
		andi	XH,0x1f
		ldi	XL,32
		mul	XL,XH
		ldd	XL,Y+4			;X
		andi	XL,0x1f
		ldi	YL,LOW(0x200)
		ldi	YH,HIGH(0x200)
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0
		ld	XL,Y
		clr	XH
		rjmp	retvalue

;-------------------------------------------------------------------------------
; set tile solid
;-------------------------------------------------------------------------------
tile_solid:	api_getvalues
		ldd	XL,Y+2			;tile
		cpi	XL,49
		brcc	tile_solid_e
		ldd	XH,Y+4			;color
		andi	XH,0x0f
		lsl	XH
		mov	tempreg1,XH
		swap	tempreg1
		or	tempreg1,XH
		ldi	XH,32
		mul	XH,XL
		ldi	YL,LOW(0x600)
		ldi	YH,HIGH(0x600)
		add	YL,r0
		adc	YH,r1
		ldi	XL,32
tile_solid_1:	st	Y+,tempreg1
		dec	XL
		brne	tile_solid_1
tile_solid_end:	rjmp	retzero

tile_solid_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

;-------------------------------------------------------------------------------
; set flash tile pattern
;-------------------------------------------------------------------------------
tile_fpattern:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_solid_e
		ldd	tempreg2,Y+4		;pattern
		andi	tempreg2,0x1f		;max 32 pattern
		ldd	tempreg3,Y+6		;color
		rcall	tile_ccolor		;convert color

		ldi	XL,LOW(ptab-start)
		ldi	XH,HIGH(ptab-start)
		api_dataptr
		ldi	XH,8			;bytes per pattern
		mul	XH,tempreg2		;*pattern nr
		add	ZL,r0
		adc	ZH,r1

		rcall	tile_taddr		;set Y to tile addres

		ldi	tempreg1,8		;Y loop counter
tile_pattern_1:	lpm	tempreg4,Z+		;get pattern data
		rcall	tile_pline
		dec	tempreg1
		brne	tile_pattern_1
		rjmp	retzero

;-------------------------------------------------------------------------------
; set tile pattern line 
; tempreg3 = color
; tempreg4 = pattern line
;-------------------------------------------------------------------------------
tile_pline:	ldi	tempreg2,4		;X loop counter
		andi	tempreg3,0x0f
tile_pline_1:	ld	XL,Y			;get tile
		sbrc	tempreg4,6
		andi	XL,0xf0			;clear second pix
		sbrc	tempreg4,6
		or	XL,tempreg3
		swap	tempreg3
		sbrc	tempreg4,7
		andi	XL,0x0f			;clear first pix
		sbrc	tempreg4,7
		or	XL,tempreg3
		swap	tempreg3
		st	Y+,XL
		lsl	tempreg4
		lsl	tempreg4
		dec	tempreg2
		brne	tile_pline_1
		ret

;-------------------------------------------------------------------------------
; set array tile pattern
;-------------------------------------------------------------------------------
tile_rpattern:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_solid_e
		ldd	tempreg2,Y+4		;pattern address
		ldd	tempreg3,Y+6		;color
		rcall	tile_ccolor		;konvert color

		api_getarray
		add	YL,tempreg2
		adc	YH,const_0
		movw	ZL,YL

		rcall	tile_taddr		;set Y to tile addres

		ldi	tempreg1,8		;Y
tile_rpatt_1:	ld	tempreg4,Z+		;get pattern data
		rcall	tile_pline
		dec	tempreg1
		brne	tile_rpatt_1
		rjmp	retzero

;-------------------------------------------------------------------------------
; set array charset pattern
;-------------------------------------------------------------------------------
tile_cpat_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

tile_cpattern:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_cpat_e
		ldd	tempreg2,Y+4		;char
		ldd	tempreg3,Y+6		;color
		rcall	tile_ccolor		;convert color
		rcall	tile_taddr		;set Y to tile addres
		api_getchart0			;set base of character table 0
		mov	ZL,tempreg2		;char offset
		inc	ZH			;skip first pixline
		ldi	r19,8			;pixlines to do

tile_cpat_01:	lpm	r16,Z			;get pixline
		mov	tempreg4,r16
		lsr	tempreg4
		ldi	tempreg1,0x3f
		ldi	tempreg2,5

tile_cpat_02:	sbrc	r16,6
		eor	tempreg4,tempreg1
		lsl	r16
		lsr	tempreg1
		dec	tempreg2
		brne	tile_cpat_02

		rcall	tile_pline

		inc	ZH			;next pixline
		dec	r19			;pixline loop counter
		brne	tile_cpat_01		;pixline loop
		rjmp	retzero			;all done

;-------------------------------------------------------------------------------
; copy tile
;-------------------------------------------------------------------------------
tile_copy:	api_getvalues
		ldd	r16,Y+2			;src tile
		cpi	r16,49
		brcc	tile_copy_e
		ldd	r17,Y+4			;src tile
		cpi	r17,49
		brcc	tile_copy_e
		rcall	tilecopy
		rjmp	retzero

tile_copy_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

;-------------------------------------------------------------------------------
; flip tile horizontal
;-------------------------------------------------------------------------------
tile_xflip:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_xflip_e
		rcall	tile_taddr
		ldi	tempreg4,8
tile_xflip_1:	ld	r16,Y
		ldd	r17,Y+3
		swap	r16
		swap	r17
		st	Y,r17
		std	Y+3,r16
		ldd	r16,Y+1
		ldd	r17,Y+2
		swap	r16
		swap	r17
		std	Y+1,r17
		std	Y+2,r16
		dec	tempreg4
		brne	tile_xflip_1
		rjmp	retzero

tile_xflip_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

;-------------------------------------------------------------------------------
; flip tile vertical
;-------------------------------------------------------------------------------
tile_yflip:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_xflip_e
		rcall	tile_taddr
		movw	ZL,YL
		adiw	ZL,28
		ldi	tempreg4,4
tile_yflip_1:	ldi	tempreg3,4
tile_yflip_2:	ld	r16,Y
		ld	r17,Z
		st	Y+,r17
		st	Z+,r16
		dec	tempreg3
		brne	tile_yflip_2
		sbiw	ZL,8
		dec	tempreg4
		brne	tile_yflip_1
		rjmp	retzero

;-------------------------------------------------------------------------------
; rotate tile clockwise
;-------------------------------------------------------------------------------
tile_cw:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_xflip_e
		rcall	tile_taddr
		ldi	ZL,0xe3			;temporary
		ldi	ZH,0x01
		movw	tempreg7,YL		;save ptr

tile_cw_2:	ldi	tempreg4,4
tile_cw_4:	ldi	tempreg3,4

tile_cw2:	ld	r16,Y+
		ldd	r17,Y+3
		movw	r18,r16
		swap	r17
		swap	r18
		andi	r19,0xf0
		andi	r17,0xf0
		andi	r18,0x0f
		andi	r16,0x0f
		or	r19,r18
		or	r17,r16
		st	Z,r19
		std	Z+4,r17

		adiw	ZL,8
		dec	tempreg3
		brne	tile_cw_4
		
		sbiw	ZL,33
		dec	tempreg4
		brne	tile_cw_2

;now copy back sprite
tile_cback:	ldi	ZL,0xe0			;temporary
		ldi	ZH,0x01
		movw	YL,tempreg7		;restore ptr
		ldi	tempreg4,32
tile_cback_1:	ld	tempreg1,Z+
		st	Y+,tempreg1
		dec	tempreg4
		brne	tile_cback_1
		rjmp	retzero

;-------------------------------------------------------------------------------
; rotate tile counterclockwise
;-------------------------------------------------------------------------------
tile_ccw:	api_getvalues
		ldd	tempreg1,Y+2		;tile
		cpi	tempreg1,49
		brcc	tile_ccw_e
		rcall	tile_taddr
		ldi	ZL,0xf8			;temporary
		ldi	ZH,0x01
		movw	tempreg7,YL		;save ptr

tile_ccw_2:	ldi	tempreg4,4
tile_ccw_4:	ldi	tempreg3,4

tile_ccw2:	ld	r16,Y+
		ldd	r17,Y+3
		movw	r18,r16
		swap	r16
		swap	r19
		andi	r19,0x0f
		andi	r17,0x0f
		andi	r18,0xf0
		andi	r16,0xf0
		or	r19,r18
		or	r17,r16
		st	Z,r17
		std	Z+4,r19

		sbiw	ZL,8
		dec	tempreg3
		brne	tile_ccw_4

		adiw	ZL,33
		dec	tempreg4
		brne	tile_ccw_2

		rjmp	tile_cback

tile_ccw_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero


;-------------------------------------------------------------------------------
; add tile information
;-------------------------------------------------------------------------------
tile_add:	rcall	tile_par2		;get 2 tile adressses
		rcall	tile_ccolor
		andi	tempreg3,0x0f
		mov	tempreg4,tempreg3
		swap	tempreg4
		ldi	tempreg2,32

tile_add_1:	ld	r16,Z			;get data 1
		ld	r17,Y			;get data 2
		andi	r16,0xf0
		andi	r17,0xf0
		cp	r17,tempreg4
		breq	tile_add_2
		mov	r16,r17
tile_add_2:	ld	r19,Z			;get data 1
		ld	r17,Y+			;get data 2
		andi	r19,0x0f
		andi	r17,0x0f
		cp	r17,tempreg3
		breq	tile_add_3
		mov	r19,r17
tile_add_3:	or	r16,r19
		st	Z+,r16
		dec	tempreg2
		brne	tile_add_1
		rjmp	retzero

;-------------------------------------------------------------------------------
; rotate 2 tiles
;-------------------------------------------------------------------------------
tile_rot:	rcall	tile_par2		;get 2 tile adressses
		andi	tempreg3,0x03
		brne	tile_rot_10		;not up 
;rotate up
		ld	tempreg5,Y
		ldd	tempreg6,Y+1
		ldd	tempreg7,Y+2
		ldd	tempreg8,Y+3
		ldi	r16,28
tile_rot_02:	ldd	tempreg1,Y+4
		st	Y+,tempreg1
		dec	r16
		brne	tile_rot_02
		ldi	r16,4
tile_rot_04:	ld	tempreg1,Z+
		st	Y+,tempreg1
		dec	r16
		brne	tile_rot_04
		sbiw	ZL,4
		ldi	r16,28
tile_rot_06:	ldd	tempreg1,Z+4
		st	Z+,tempreg1
		dec	r16
		brne	tile_rot_06
		st	Z+,tempreg5
		st	Z+,tempreg6
		st	Z+,tempreg7
		st	Z+,tempreg8
		rjmp	retzero
;rotate down
tile_rot_10:	cpi	tempreg3,0x02		;down
		brne	tile_rot_20		;not down 
		adiw	ZL,32
		adiw	YL,32
		ld	tempreg8,-Z
		ld	tempreg7,-Z
		ld	tempreg6,-Z
		ld	tempreg5,-Z
		ldi	r16,28
tile_rot_12:	ld	tempreg1,-Z
		std	Z+4,tempreg1
		dec	r16
		brne	tile_rot_12
		ldi	r16,4
		adiw	ZL,4
tile_rot_14:	ld	tempreg1,-Y
		st	-Z,tempreg1
		dec	r16
		brne	tile_rot_14
		sbiw	ZL,4
		ldi	r16,28
tile_rot_16:	ld	tempreg1,-Y
		std	Y+4,tempreg1
		dec	r16
		brne	tile_rot_16
		st	Y,tempreg5
		std	Y+1,tempreg6
		std	Y+2,tempreg7
		std	Y+3,tempreg8
		rjmp	retzero
;rotate right
tile_rot_20:	cpi	tempreg3,0x01		;right
		brne	tile_rot_30		;not right
		ldi	r16,8
tile_rot_22:	ldd	tempreg2,Z+3		;save
		ldd	tempreg1,Z+2
		std	Z+3,tempreg1
		ldd	tempreg1,Z+1
		std	Z+2,tempreg1
		ld	tempreg1,Z
		std	Z+1,tempreg1
		ldd	tempreg1,Y+3
		st	Z,tempreg1
		ldd	tempreg1,Y+2
		std	Y+3,tempreg1
		ldd	tempreg1,Y+1
		std	Y+2,tempreg1
		ld	tempreg1,Y
		std	Y+1,tempreg1
		st	Y,tempreg2
		adiw	YL,4
		adiw	ZL,4
		dec	r16
		brne	tile_rot_22
		rjmp	retzero
;rotate left
tile_rot_30:	ldi	r16,8
tile_rot_32:	ld	tempreg2,Y		;save
		ldd	tempreg1,Y+1
		st	Y,tempreg1
		ldd	tempreg1,Y+2
		std	Y+1,tempreg1
		ldd	tempreg1,Y+3
		std	Y+2,tempreg1
		ld	tempreg1,Z
		std	Y+3,tempreg1
		ldd	tempreg1,Z+1
		st	Z,tempreg1
		ldd	tempreg1,Z+2
		std	Z+1,tempreg1
		ldd	tempreg1,Z+3
		std	Z+2,tempreg1
		std	Z+3,tempreg2
		adiw	YL,4
		adiw	ZL,4
		dec	r16
		brne	tile_rot_32
		rjmp	retzero

;-------------------------------------------------------------------------------
;get tile address
;tempreg1 = tile number
;-------------------------------------------------------------------------------
tile_taddr:	push	tempreg4
		ldi	tempreg4,32
		mul	tempreg4,tempreg1
		movw	YL,r0
		subi	YH,0xfa
		pop	tempreg4
		ret

;-------------------------------------------------------------------------------
; set color for 8C/16C
; tempreg3=color (in/out)
;-------------------------------------------------------------------------------
tile_ccolor:	push	tempreg4
		mov	tempreg4,tempreg3
		andi	tempreg4,0x88
		andi	tempreg3,0x77
		lsl	tempreg3
		lsr	tempreg4
		lsr	tempreg4
		lsr	tempreg4
		or	tempreg3,tempreg4
		pop	tempreg4
		ret

;-------------------------------------------------------------------------------
; get 2 tile adr (Y,Z) and 1 parameter in tempreg3
; tempreg3=color (in/out)
;-------------------------------------------------------------------------------
tile_par2_e:	pop	r16			;kill stack
		pop	r16
		ldi	ereg,5			;constant out of range
		rjmp	retzero

tile_par2:	api_getvalues
		ldd	tempreg1,Y+2		;src/dest tile
		ldd	tempreg2,Y+4		;src tile 2
		ldd	tempreg3,Y+6		;transp. color
		cpi	tempreg1,49
		brcc	tile_par2_e
		cpi	tempreg2,49
		brcc	tile_par2_e
		rcall	tile_taddr		;Z= t1 address
		movw	ZL,YL
		mov	tempreg1,tempreg2
		rjmp	tile_taddr		;Y= t2 address

;-------------------------------------------------------------------------------
; draw sprite
; par1=array adr
; par2=new y
; par3=new x
;-------------------------------------------------------------------------------
ti_spdraw:	api_getvalues
		ldd	XL,Y+2			;array pos
		ldd	XH,Y+3
		cpi	XH,2
		brcs	ti_spdraw_02
		cpi	XL,0xf1
		brcs	ti_spdraw_02
		ldi	ereg,18			;out of array
		ret

ti_spdraw_02:	ldd	tempreg2,Y+4		;new y
		ldd	tempreg3,Y+6		;new x
		api_getarray
		add	YL,XL
		adc	YH,XH
		ldd	tempreg1,Y+7		;visible flag
		sbrc	tempreg1,0
		rcall	ti_restore
		st	Y,tempreg2		;store new coordinates
		std	Y+1,tempreg3
		std	Y+7,const_1		;set as visible
		rcall	ti_backup
		rcall	ti_draw
		rjmp	retzero

;------------------------------------------------------------------------------
;backup tiles for sprite struct Y
;+0	Ynew	Byte
;+1	Xnew	Byte
;+2	Yold	Byte
;+3	Xold	Byte
;+4	Sprite tile	Byte
;+5	bstart tile 	Byte
;+6	tcolor	Byte
;+7	visible (0/1)
;+8	active (0/1) for sequential drawing
;+9	collision flag
;+a	Y step
;+b	X step
;+c	tile number at pos Y0X0
;+d	tile number at pos Y0X1
;+e	tile number at pos Y1X0
;+f	tile number at pos Y1X1
;------------------------------------------------------------------------------
ti_backup:	ld	XH,Y			;Ynew
		std	Y+2,XH			;save as old
		ldd	XL,Y+1			;Xnew
		std	Y+3,XL			;save as old
		rcall	ti_calcpos		;calculate sprite pos and size
		ldd	r17,Y+5			;bstart
		ldi	tempreg3,32		;tile mem size
;calculate bstart tile address
		mul	r17,tempreg3
		movw	XL,r0
		subi	XH,0xfa			;add tiledata start 0x600

;copy tile Y0X0
		ld	r16,Z			;tile 0
		std	Y+12,r16
		st	Z,r17			;set this pos to tile bstart+0
		rcall	ti_btile		;copy tile Y+0,X+0

;copy tile Y0X1
		inc	r17			;next backup tile
		push	ZL			;save pos
		mov	r19,ZL			;copy pos
		andi	r19,0xe0
		inc	ZL
		andi	ZL,0x1f
		or	ZL,r19
		ld	r16,Z			;tile 1
		std	Y+13,r16
		st	Z,r17			;set this pos to tile bstart+1
		rcall	ti_btile		;copy tile Y+0,X+1
		pop	ZL

;copy tile Y1X0
		inc	r17			;next backup tile
		adiw	ZL,32
		cpi	ZH,0x06
		brcs	ti_backup_5
		subi	ZH,0x04
ti_backup_5:
		ld	r16,Z			;tile 0
		std	Y+14,r16
		st	Z,r17			;set this pos to tile bstart+0
		rcall	ti_btile		;copy tile Y+0,X+0

;copy tile Y1X1
		inc	r17			;next backup tile
		mov	r19,ZL			;copy pos
		andi	r19,0xe0
		inc	ZL
		andi	ZL,0x1f
		or	ZL,r19
		ld	r16,Z			;tile 1
		std	Y+15,r16
		st	Z,r17			;set this pos to tile bstart+1
		rcall	ti_btile		;copy tile Y+0,X+1
		ret

;copy tile data from tile r16 to X+
ti_btile:	movw	tempreg5,YL
		mul	r16,tempreg3
		movw	YL,r0
		subi	YH,0xfa			;add tiledata start 0x600
		ldi	tempreg2,8
ti_btile_1:	ld	tempreg1,Y+
		st	X+,tempreg1
		ld	tempreg1,Y+
		st	X+,tempreg1
		ld	tempreg1,Y+
		st	X+,tempreg1
		ld	tempreg1,Y+
		st	X+,tempreg1
		dec	tempreg2
		brne	ti_btile_1
		movw	YL,tempreg5
		ret

;------------------------------------------------------------------------------
;draw tiles for sprite struct Y
;+0	Ynew	Byte
;+1	Xnew	Byte
;+2	Yold	Byte
;+3	Xold	Byte
;+4	Sprite tile	Byte
;+5	bstart tile 	Byte
;+6	tcolor	Byte
;+7	visible (0/1)
;+8	active (0/1) for sequential drawing
;+9	collision flag
;+a	Y step
;+b	X step
;+c	tile number at pos Y0X0
;+d	tile number at pos Y0X1
;+e	tile number at pos Y1X0
;+f	tile number at pos Y1X1
;------------------------------------------------------------------------------
ti_draw:
		ldd	ZH,Y+4			;sprite tile
		ldd	tempreg3,Y+6		;tcolor
		rcall	tile_ccolor
		mov	r18,tempreg3		;save transparent color
		ldi	tempreg3,32		;bytes per tile (used as constant)
		mul	tempreg3,ZH		;sprite tile * tile size
		movw	ZL,r0
		subi	ZH,0xfa			;calc sprite tile address to Z
;Z is now sprite tile address
		ldd	XH,Y+5			;backup tile
		mul	tempreg3,XH		;backup tile * tile size
		movw	YL,r0
		subi	YH,0xfa			;calc backup tile address to Y
		movw	tempreg5,YL		;save backup tile address
;Y is now backup tile address

;set r19 to vertical offset
		mov	r19,tempreg4		;copy offset byte
		swap	r19			;we need upper nibble
		andi	r19,0x07		;mask bits

;set tempreg4 to horizontal offset
		andi	tempreg4,0x07		;mask bits

		ldi	tempreg2,8		;pixlines to do

ti_draw_10:	ldi	tempreg1,8		;pixel to do

ti_draw_12:	ld	r17,Z			;get sprite data
		sbrs	tempreg1,0		;skip if second nibble
		swap	r17			;swap data
		sbrc	tempreg1,0		;skip if first nibble
		adiw	ZL,1			;prepare next byte
		andi	r17,0x0f		;mask data to write
		cp	r17,r18			;check if transparent color
		breq	ti_draw_14		;branch if transparent
;r17/0..3 is now source nibble

		mov	YH,r19			;vertical offset
		lsl	YH			;*4 (4 bytes per pixline)
		lsl	YH
		bst	YH,5			;tile select vertical
		bld	YH,6			;store to new position
		andi	YH,0x5c			;mask bits

		mov	YL,tempreg4		;horizontal offset
		lsr	YL			;shift right
		bst	YL,2			;tile select horizontal
		bld	YL,5			;store to new position
		andi	YL,0x23			;mask bits
		or	YL,YH
		clr	YH
		add	YL,tempreg5
		adc	YH,tempreg6

		ldi	XH,0xf0			;bit bmask
		sbrs	tempreg4,0		;skip if lower nibble is to be replaced
		swap	XH			;swap bit mask

		sbrs	tempreg4,0		;skip if upper nibble is to be replaced
		swap	r17			;swap source nibble

		ld	XL,Y			;get data
		and	XL,XH			;mask bits
		or	XL,r17			;replace bits
		st	Y,XL			;store back data

ti_draw_14:	inc	tempreg4
		dec	tempreg1
		brne	ti_draw_12

		subi	tempreg4,8
		inc	r19
		dec	tempreg2
		brne	ti_draw_10
		ret

;-------------------------------------------------------------------------------
; undraw sprite
; par1=array adr
;-------------------------------------------------------------------------------
ti_undraw:	api_getvalues
		ldd	XL,Y+2			;array pos
		ldd	XH,Y+3
		cpi	XH,2
		brcs	ti_undraw_02
		cpi	XL,0xf9
		brcs	ti_undraw_02
		ldi	ereg,18			;out of array
		ret

ti_undraw_02:	api_getarray
		add	YL,XL
		adc	YH,XH
		ldd	tempreg1,Y+7		;visible flag
		sbrc	tempreg1,0
		rcall	ti_restore
		std	Y+7,const_0		;clear visible flag
		rjmp	retzero


;------------------------------------------------------------------------------
;calc addresses of tile Y(n)X(m) and Y(n)X(m+1)
;tempreg5/6 = address of tile YnXm
;tempreg7/8 = address of tile YnXm+1
;------------------------------------------------------------------------------
;calc addresss of tile Y0X0
ti_calc2:	ld	r16,X			;get tile X=0
		mul	r16,tempreg3		;* tile size
		movw	YL,r0
		subi	YH,0xfa			;+0x600 tile data base
		movw	tempreg5,YL		;tile address Y0X0

;calc addresss of tile Y0X1
		push	XL
		mov	YL,XL
		andi	YL,0xe0
		inc	XL
		andi	XL,0x1f
		or	XL,YL
		ld	r16,X			;get tile X=0
		mul	r16,tempreg3
		movw	YL,r0
		subi	YH,0xfa			;+0x600 tile data base
		movw	tempreg7,YL		;tile address X0Y0
		pop	XL
		ret

;------------------------------------------------------------------------------
;draw tiles for sprite struct Y
;+0	Ynew	Byte
;+1	Xnew	Byte
;+2	Yold	Byte
;+3	Xold	Byte
;+4	Sprite tile	Byte
;+5	bstart tile 	Byte
;+6	tcolor	Byte
;+7	visible (0/1)
;+8	active (0/1) for sequential drawing
;+9	collision flag
;+a	Y step
;+b	X step
;+c	tile number at pos Y0X0
;+d	tile number at pos Y0X1
;+e	tile number at pos Y1X0
;+f	tile number at pos Y1X1
;------------------------------------------------------------------------------
ti_restore:	ldd	XH,Y+2			;Yold
		ldd	XL,Y+3			;Xold
		rcall	ti_calcpos		;calculate sprite pos and size

;restore tile Y0X0
		ldd	r16,Y+12
		st	Z,r16			;tile 0

;restore tile Y0X1
		mov	XL,ZL			;save pos
		mov	XH,ZL			;copy pos
		andi	XH,0xe0
		inc	ZL
		andi	ZL,0x1f
		or	ZL,XH
		ldd	r16,Y+13
		st	Z,r16			;tile 1
		mov	ZL,XL

;restore tile Y1X0
		adiw	ZL,32
		cpi	ZH,0x06
		brcs	ti_restore_1
		subi	ZH,0x04
ti_restore_1:
		ldd	r16,Y+14
		st	Z,r16			;tile 2

;restore tile Y1X1
		mov	XH,ZL			;copy pos
		andi	XH,0xe0
		inc	ZL
		andi	ZL,0x1f
		or	ZL,XH
		ldd	r16,Y+15
		st	Z,r16			;tile 3
		ret

;------------------------------------------------------------------------------
; calc map position
; XL=Xpos
; XH=Ypos
; > Z=map position
; > tempreg4 = pixel pos
; bit 6..4	vertical
; bit 2..0	horizontal
;------------------------------------------------------------------------------
ti_calcpos:	mov	tempreg4,XL		;offset
		andi	tempreg4,0x07
		swap	tempreg4
		lsr	XL			;X pos/8
		lsr	XL
		lsr	XL
		lsr	XH			;y-pos/2
		ror	tempreg4		;shift bit into pixel pos
		lsr	XH			;y-pos/2
		ror	tempreg4		;shift bit into pixel pos
		lsr	XH			;y-pos/2
		ror	tempreg4		;shift bit into pixel pos
		lsr	tempreg4		;one additional shift
		ldi	ZL,32
		mul	ZL,XH
		movw	ZL,r0			;Y offset
		add	ZL,XL			;X offset
		subi	ZH,0xfe			;map offset
		ret

;------------------------------------------------------------------------------
; set sprite start and number
;------------------------------------------------------------------------------
tile_spinit:	api_getvalues
		ldd	XL,Y+2			;array pos
		ldd	XH,Y+3
		andi	XH,0x01
		ldd	r16,Y+4			;number
		dec	r16
		andi	r16,0x07		;max 8 sprites
		api_getarray
		add	YL,XL
		adc	YH,XH
		sts	0x1dc,YL		;store ptr
		sts	0x1dd,YH
		inc	r16
		sts	0x1de,r16		;store num
		ldi	tempreg1,45		;backup tiles
tile_spinit_1:	st	Y+,const_0		;+0	Ynew	Byte
		st	Y+,const_0		;+1	Xnew	Byte
		st	Y+,const_0		;+2	Yold	Byte
		st	Y+,const_0		;+3	Xold	Byte
		st	Y+,const_0		;+4	Sprite tile	Byte
		st	Y+,tempreg1		;+5	bstart tile 	Byte
		st	Y+,const_0		;+6	tcolor	Byte
		st	Y+,const_0		;+7	visible (0/1)
		st	Y+,const_0		;+8	active (0/1) for sequential drawing
		st	Y+,const_0		;+9	collision flag
		st	Y+,const_0		;+a	Y step
		st	Y+,const_0		;+b	X step
		st	Y+,const_0		;+c	tile number at pos Y0X0
		st	Y+,const_0		;+d	tile number at pos Y0X1
		st	Y+,const_0		;+e	tile number at pos Y1X0
		st	Y+,const_0		;+f	tile number at pos Y1X1

		subi	tempreg1,4		;4 backup tiles per sprite
		dec	r16
		brne	tile_spinit_1
		subi	tempreg1,0xfd		;+3
		mov	XL,tempreg1
		clr	XH
		rjmp	retvalue		;return max sprite nr

;------------------------------------------------------------------------------
; undraw sprite seq
;------------------------------------------------------------------------------
tile_useq:	rcall	tile_gap
		dec	r16
		ldi	tempreg1,16
		mul	tempreg1,r16
		add	YL,r0
		adc	YH,r1
		inc	r16

tile_useq_1:	ldd	tempreg1,Y+7		;visible flag
		cpi	tempreg1,0
		breq	tile_useq_4		;not active
		push	YL
		push	YH
		push	r16
		rcall	ti_restore
		pop	r16
		pop	YH
		pop	YL
		std	Y+7,const_0		;set as visible
tile_useq_4:	sbiw	YL,16
		dec	r16
		brne	tile_useq_1
		rjmp	retzero

;------------------------------------------------------------------------------
; get array pointer
;------------------------------------------------------------------------------
tile_gap:	lds	YL,0x1dc		;get array ptr
		lds	YH,0x1dd
		lds	r16,0x1de		;get num
		ret

;------------------------------------------------------------------------------
; draw sprite seq
;------------------------------------------------------------------------------
tile_dseq:	rcall	tile_gap		;get array ptr

tile_dseq_1:	ldd	tempreg1,Y+8		;active flag
		cpi	tempreg1,0
		breq	tile_dseq_4		;not active
		std	Y+7,const_1		;set as visible
		push	YL
		push	YH
		push	r16
		rcall	ti_backup
		rcall	ti_draw
		pop	r16
		pop	YH
		pop	YL
tile_dseq_4:	adiw	YL,16
		dec	r16
		brne	tile_dseq_1
		rjmp	retzero

;------------------------------------------------------------------------------
; set sprite type
;------------------------------------------------------------------------------
tile_stype:	api_getvalues
		ldd	XL,Y+2			;sprite no
		ldd	XH,Y+4			;tile_no
		ldd	tempreg2,Y+6		;transp. color
		rcall	tile_gap		;get array ptr
		cp	XL,r16
		brcc	tile_stype_e
		cpi	XH,49
		brcc	tile_stype_e
		ldi	tempreg1,16
		mul	tempreg1,XL
		add	YL,r0
		adc	YH,r1
		std	Y+4,XH			;set sprite tile
		std	Y+8,const_1		;activate
		std	Y+6,tempreg2		;transp color
		rjmp	retzero

tile_stype_e:	ldi	ereg,5			;constant out of range
		rjmp	retzero

;------------------------------------------------------------------------------
; set sprite coords
;------------------------------------------------------------------------------
tile_scoords:	api_getvalues
		ldd	XL,Y+2			;sprite no
		ldd	XH,Y+4			;Y
		ldd	tempreg2,Y+6		;X
		ldd	tempreg3,Y+8		;dy
		ldd	tempreg4,Y+10		;dx
		rcall	tile_gap		;get array ptr
		cp	XL,r16
		brcc	tile_stype_e
		ldi	tempreg1,16
		mul	tempreg1,XL
		add	YL,r0
		adc	YH,r1
		std	Y+0,XH
		std	Y+1,tempreg2
		std	Y+10,tempreg3
		std	Y+11,tempreg4
		rjmp	retzero

;------------------------------------------------------------------------------
; disable sprite in sequence
;------------------------------------------------------------------------------
tile_sdisable:	api_getvalues
		ldd	XL,Y+2			;sprite no
		rcall	tile_gap		;get array ptr
		cp	XL,r16
		brcc	tile_stype_e
		ldi	tempreg1,16
		dec	XL
		mul	tempreg1,XL
		add	YL,r0
		adc	YH,r1
		std	Y+8,const_0		;de-activate
		rjmp	retzero

;------------------------------------------------------------------------------
; enable sprite in sequence
;------------------------------------------------------------------------------
tile_senable:	api_getvalues
		ldd	XL,Y+2			;sprite no
		rcall	tile_gap		;get array ptr
		cp	XL,r16
		brcc	tile_stype_e
		ldi	tempreg1,16
		dec	XL
		mul	tempreg1,XL
		add	YL,r0
		adc	YH,r1
		std	Y+8,const_1		;activate
		rjmp	retzero

;------------------------------------------------------------------------------
; move all sprites
;------------------------------------------------------------------------------
tile_movea:	rcall	tile_gap		;get array ptr

tile_movea_1:	ldd	tempreg1,Y+8		;active flag
		cpi	tempreg1,0
		breq	tile_movea_4		;not active
		push	YL
		push	YH
		push	r16
;calc new Y
		ldd	tempreg2,Y+2
		ldd	tempreg3,Y+10
		add	tempreg2,tempreg3
		st	Y,tempreg2

;calc new X
		ldd	tempreg2,Y+3
		ldd	tempreg3,Y+11
		add	tempreg2,tempreg3
		std	Y+1,tempreg2

		pop	r16
		pop	YH
		pop	YL
tile_movea_4:	adiw	YL,16
		dec	r16
		brne	tile_movea_1
		rjmp	retzero

;------------------------------------------------------------------------------
; undraw-move-draw all sprites
;------------------------------------------------------------------------------
tile_movea_drw:	ldi	tempreg1,223
tile_movea_d1:	cp	vline_l,tempreg1
		brne	tile_movea_d1
		rcall	tile_useq
		rcall	tile_movea
		rjmp	tile_dseq

;------------------------------------------------------------------------------
; copy tile R16 to tile R17
;------------------------------------------------------------------------------
tilecopy:	api_pushxyz
		ldi	XH,32			;bytes per tile
		mul	r16,XH
		movw	YL,r0
		subi	YH,0xfa			;+0x600 tile data base
		mul	r17,XH
		movw	ZL,r0
		subi	ZH,0xfa			;+0x600 tile data base
tilecopy_1:	ld	XL,Y+
		st	Z+,XL
		dec	XH
		brne	tilecopy_1
		api_popxyz

;-------------------------------------------------------------------------------
; pattern table 1
;-------------------------------------------------------------------------------
ptab:		.db	0xaa,0x55,0xaa,0x55,0xaa,0x55,0xaa,0x55		;00 pepper and salt
		.db	0xff,0x80,0x80,0x80,0xff,0x08,0x08,0x08		;01 wall
		.db	0x3c,0x7e,0xff,0xff,0xff,0xff,0x7e,0x3c		;02 ball
		.db	0x3c,0x42,0x81,0x81,0x81,0x81,0x42,0x3c		;03 ring
		.db	0x18,0x3c,0x7e,0xff,0xff,0x7e,0x3c,0x18		;04 rhombus
		.db	0xff,0x81,0x81,0x81,0x81,0x81,0x81,0xff		;05 rect
		.db	0xff,0xc3,0xa5,0x99,0x99,0xa5,0xc3,0xff		;06 crossed rect
		.db	0x66,0xff,0x7e,0x7e,0x3c,0x3c,0x18,0x18		;07 heart
		.db	0x65,0x94,0xe1,0x26,0x48,0x64,0x92,0x4c		;08 random 1
		.db	0x3c,0xff,0xbd,0xff,0xc3,0xbd,0xff,0xdb		;09 ghost 1 
		.db	0x3c,0xff,0xbd,0xff,0xbd,0xff,0xff,0xdb		;0a ghost 2
		.db	0x3c,0x7e,0xff,0xff,0x18,0x18,0x18,0x18		;0B mushroom
		.db	0x3c,0x7e,0xff,0x7e,0x3c,0x18,0x18,0x3c		;0C tree 1
		.db	0x18,0x18,0x3c,0x3c,0x7e,0x7e,0x18,0x18		;0D tree 2
		.db	0x7e,0xff,0xff,0xff,0xff,0xff,0xff,0x7e		;0E rect
		.db	0x18,0x18,0x3c,0x3c,0x7e,0x7e,0xff,0xff		;0F triangle

		.db	0x3c,0x7e,0xbd,0xe7,0xff,0xc3,0x7e,0x3c		;10 smile 1
		.db	0x3c,0x7e,0xbd,0xe7,0xbd,0xc3,0x7e,0x3c		;11 smile 2
		.db	0x3c,0x3c,0x18,0x3c,0xff,0x3c,0x24,0x24		;12 man 1
		.db	0x3c,0x3c,0x18,0x3c,0xff,0x3c,0x24,0xc3		;13 man 2
		.db	0x99,0xff,0xbd,0x3c,0x3c,0xbd,0xff,0xbd		;14 car 1
		.db	0x01,0x03,0x07,0x0f,0x1f,0x3f,0x7f,0xff		;15 diag 1
		.db	0x00,0x00,0x7e,0x7e,0x7e,0x7e,0x00,0x00		;16 brick 1
		.db	0x18,0x3c,0x3c,0x3c,0x3c,0x7e,0xff,0xbd		;17 rocket
		.db	0x3c,0x3c,0xdb,0xff,0xff,0xdb,0x3c,0x3c		;18 flower/cross
		.db	0x18,0x3c,0x7e,0xdb,0x18,0x18,0x18,0x18		;19 arrow 1
		.db	0x18,0x3c,0x7e,0xff,0x18,0x18,0x18,0x18		;1a arrow 2
		.db	0x18,0x18,0x18,0xff,0xff,0x18,0x18,0x18		;1b crosshair 1
		.db	0x10,0x10,0x10,0xfe,0x10,0x10,0x10,0x00		;1c crosshair 2
		.db	0xfe,0x92,0x92,0xfe,0x92,0x92,0xfe,0x00		;1d crosshair 3
		.db	0x18,0x3c,0x7e,0x7e,0x7e,0x7e,0xff,0x18		;1e bell
		.db	0xdb,0x3c,0xbd,0x7e,0x7e,0xbd,0x24,0xc3		;1f spider

