;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# 8080 emulator for CP/M	 						#
;# copyright (c) 2006-2010 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 3		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

;The hw is modelled to make writing a CPM BIOS easier.
;Ports:
;IN(x)
;0 - Con status. Returns 0xFF if the UART has a byte, 0 otherwise.
;1 - Console input, aka UDR.
;OUT(X)
;2 - Console output
;16 - Track select
;18 - Sector select
;20 - Write addr l
;21 - Write addr h
;22 - Trigger - write 1 to read, 2 to write a sector using the above info.
;	This will automatically move track, sector and dma addr to the next sector.


.def		const_0		= r2
.def		const_1		= r3
.def		const_ff	= r4
.def		xram_strobe	= r5
.def		xram_busy	= r6
.def		jtab_hi		= r7
.def		reg_c		= r12
.def		reg_b		= r13
.def		reg_e		= r14
.def		reg_d		= r15


.def		r_datl		= r16
.def		r_dath		= r17
.def		reg_a		= r20
.def		reg_f		= r21
.def		reg_l		= r22
.def		reg_h		= r23
.def		PC_L		= r24
.def		PC_H		= r25

.def		emu_stackl	= r0		;stack pointer (L)
.def		emu_stackh	= r1		;stack pointer (H)

.CSEG
.include	"/usr/local/include/avr/m644Pdef.inc"
.include	"../../../system/includes/api_macros.inc"
;.include	"../../../system/includes/libdef.asm"
.include	"../../../system/includes/definitions.asm"

.org	0x4800
;                   "......      N"  
start:		.db " EMU-  8080 N",0xec,0xff,0x00
;icon def
		.db 0x07,0x03,0x03,0x0b
		.db "8080"
		.db 0x0d,0x0c,0x0c,0x0e

;dummy data
		.db 0x00,0x00,0x00,0x00

.org	0x4810
		ldi	YL,0x88
		ldi	YH,0x01
		st	Y+,reg_a
		st	Y+,reg_f
		st	Y+,reg_b
		st	Y+,reg_c
		st	Y+,reg_d
		st	Y+,reg_e
		st	Y+,reg_h
		st	Y+,reg_l
		st	Y+,PC_L
		st	Y+,PC_H

		rcall	emu_start
		sts	0xcb8,PC_L
		sts	0xcba,PC_H
		sts	0xca0,reg_a
		sts	0xca2,reg_b
		sts	0xca4,reg_c
		sts	0xca6,reg_d
		sts	0xca8,reg_e
		sts	0xcaa,reg_f
		sts	0xcae,reg_h
		sts	0xcb6,reg_l
		sts	0xcbe,r0		;SPL
		sts	0xcc0,r1		;SPH

		ldi	YL,0x88
		ldi	YH,0x01
		ld	reg_a,Y+
		ld	reg_f,Y+
		ld	reg_b,Y+
		ld	reg_c,Y+
		ld	reg_d,Y+
		ld	reg_e,Y+
		ld	reg_h,Y+
		ld	reg_l,Y+
		ld	PC_L,Y+
		ld	PC_H,Y+
		ret

;------------------------------------------------------------------------------
; get next byte from PC+ to ZL
; 5 words, 6 clocks
;------------------------------------------------------------------------------
.macro emu_nextbyte
		out	PINB,xram_busy		;1
		adiw	PC_L,1			;2
		nop
		in	ZL,PINA			;1
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; get next word from PC+ to XH/XL
; 9 words, 11 clocks
;------------------------------------------------------------------------------
.macro emu_nextword
		out	PINB,xram_busy		;1
		adiw	PC_L,1			;2
		nop
		in	XL,PINA			;1 get low byte
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
		adiw	PC_L,1			;2
		nop
		in	XH,PINA			;1 get high byte
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; set mem addr to PC and prepare READ
; 8 words, 8 clocks
;------------------------------------------------------------------------------
.macro emu_pcaddr_read
		out	DDRA,const_ff		;1 output
		out	PORTA,const_1		;1 select read
		out	PINB,xram_strobe	;1 latch HI address
		out	PORTA,PC_H		;1 set HI address
		out	PINB,xram_busy		;1
		out	PORTA,PC_L		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_strobe	;1 set new address
		out	DDRA,const_0		;1 input
.endmacro

;------------------------------------------------------------------------------
; set mem addr to X and prepare READ
; 8 words, 8 clocks
;------------------------------------------------------------------------------
.macro emu_X_read
		out	DDRA,const_ff		;1 output
		out	PORTA,const_1		;1 select read
		out	PINB,xram_strobe	;1 latch HI address
		out	PORTA,XH		;1 set HI address
		out	PINB,xram_busy		;1
		out	PORTA,XL		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_strobe	;1 set new address
		out	DDRA,const_0		;1 input
.endmacro

;------------------------------------------------------------------------------
; set mem addr to X and prepare READ
; 8 words, 8 clocks
;------------------------------------------------------------------------------
.macro emu_SP_read
		out	DDRA,const_ff		;1 output
		out	PORTA,const_1		;1 select read
		out	PINB,xram_strobe	;1 latch HI address
		out	PORTA,r1		;1 set HI address
		out	PINB,xram_busy		;1
		out	PORTA,r0		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_strobe	;1 set new address
		out	DDRA,const_0		;1 input
		out	PINB,xram_busy		;1
		add	r0,const_1		;1
		adc	r1,const_0		;1
		in	r_datl,PINA		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
		add	r0,const_1		;1
		adc	r1,const_0		;1
		in	r_dath,PINA		;1
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; set mem addr to PC and prepare WRITE
; 7 words, 7 clocks
;------------------------------------------------------------------------------
.macro emu_X_write
		out	DDRA,const_ff		;1 output
		out	PORTA,const_0		;1 select write
		out	PINB,xram_strobe	;1 latch HI address
		out	PORTA,XH		;1 set HI address
		out	PINB,xram_busy		;1
		out	PORTA,XL		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_strobe	;1 set new address
.endmacro

;------------------------------------------------------------------------------
; set mem addr to PC and prepare WRITE
; 7 words, 7 clocks
;------------------------------------------------------------------------------
.macro emu_SP_write
		sub	r0,const_1
		sbc	r1,const_0
		sub	r0,const_1
		sbc	r1,const_0
		out	DDRA,const_ff		;1 output
		out	PORTA,const_0		;1 select write
		out	PINB,xram_strobe	;1 latch HI address
		out	PORTA,r1		;1 set HI address
		out	PINB,xram_busy		;1
		out	PORTA,r0		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_strobe	;1 set new address
		out	PORTA,r_datl		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
		out	PORTA,r_dath		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; read r_datl from (X)
; 9 words, 9 clocks
;------------------------------------------------------------------------------
.macro emu_rbyte
		emu_X_read
		out	PINB,xram_busy		;1
		nop				;1
		nop				;1
		out	PINB,xram_busy		;1
		in	r_datl,PINA		;1
.endmacro

;------------------------------------------------------------------------------
; read r_dat from (X)
; 13 words, 13 clocks
;------------------------------------------------------------------------------
.macro emu_rword
		emu_X_read
		out	PINB,xram_busy		;1
		nop				;1
		nop				;1
		in	r_datl,PINA		;1
		out	PINB,xram_busy		;1
		nop
		out	PINB,xram_busy		;1
		nop				;1
		nop				;1
		in	r_dath,PINA		;1
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; write r_datl to (X)
; 8 words, 8 clocks
;------------------------------------------------------------------------------
.macro emu_wbyte
		emu_X_write
		out	PORTA,r_datl		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
.endmacro

;------------------------------------------------------------------------------
; write r_dath/r_datl to (X)
; 11 words, 11 clocks
;------------------------------------------------------------------------------
.macro emu_wword
		emu_X_write
		out	PORTA,r_datl		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
		out	PORTA,r_dath		;1
		out	PINB,xram_busy		;1
		out	PINB,xram_busy		;1
.endmacro


;##############################################################################
; init and start the VM
;##############################################################################
emu_start:	api_getbase
		subi	ZH,0xfe
		mov	jtab_hi,ZH
		ldi	XL,0xFF
		mov	const_ff,XL
		ldi	XL,0x02
		mov	xram_strobe,XL
		ldi	XL,0x04
		mov	xram_busy,XL
		mov	r0,const_0		;init SP
		mov	r1,const_0
		clr	PC_L			;PC
		clr	PC_H			;PC
		clr	reg_a			;A
		clr	reg_f			;F
		mov	reg_b,const_0
		mov	reg_c,const_0
		mov	reg_d,const_0
		mov	reg_e,const_0
		mov	reg_h,const_0
		mov	reg_l,const_0
		ldi	YH,0			;1 high register pointer
		rjmp	emu_loop		;now start

;------------------------------------------------------------------------------
; get (rr) to r_datl
;------------------------------------------------------------------------------
emu_gethl:	movw	XL,reg_l
		emu_rbyte
		ret

;------------------------------------------------------------------------------
; put r_datl to (rr)
;------------------------------------------------------------------------------
emu_puthl:	movw	XL,reg_l
		emu_wbyte
		rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; write next word to 16 bit reg
;------------------------------------------------------------------------------
emu_regset16:	emu_nextword			;9
		st	Y+,XL			;2 store L
		std	Y+1,XH			;2 store H
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; write accu to addr (Y)
;------------------------------------------------------------------------------
emu_astore:	mov	r_datl,reg_a		;1
;write byte from r_dath
emu_bstore:	ld	XL,Y			;register LO
		ldd	XH,Y+1			;register HI
emu_cstore:	emu_wbyte			;8
		rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; read accu from addr (Y)
;
;------------------------------------------------------------------------------
emu_aread:	ld	XL,Y			;register LO
		ldd	XH,Y+1			;register HI	
emu_areadx:	emu_rbyte			;8
		mov	reg_a,r_datl		;1
		rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; inc (Y) register
;------------------------------------------------------------------------------
emu_incb:	ld	r_datl,Y		;2 get register
		bst	reg_f,0			;1 save carry flag
		add	r_datl,const_1		;1
		in	reg_f,SREG		;1
		bld	reg_f,0			;1 restore carry flag
		st	Y,r_datl		;2 put back register
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; dec (Y) register
;------------------------------------------------------------------------------
emu_decb:	ld	r_datl,Y		;2 get register
		bst	reg_f,0			;1 save carry flag
		sub	r_datl,const_1		;1
		in	reg_f,SREG		;1
		bld	reg_f,0			;1 restore carry flag
		st	Y,r_datl		;2 put back register
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; set (Y) register immediate byte from (PC+)
;------------------------------------------------------------------------------
emu_setib:	emu_nextbyte			;1 get next byte
		st	Y,ZL
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; set (Y) word immediate word from (PC+)
;------------------------------------------------------------------------------
emu_setiw:	emu_nextword			;1 get next byte
emu_stor16:	st	Y,XL			;2 store LO
		std	Y+1,XH			;2 store HI
		clr	YH
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; add 16 bit value (Y) to HL
;------------------------------------------------------------------------------
emu_add16:	ld	r_datl,Y		;2 get HI
		ldd	r_dath,Y+1		;2 get LO
		add	reg_l,r_datl		;1 add
		adc	reg_h,r_dath		;1
		in	r_datl,SREG		;1 clear
		bst	r_datl,0		;1 copy carry
		bld	reg_f,0			;1
		rjmp	emu_nloop		;2 goto loop

;------------------------------------------------------------------------------
; jump absolute if T is set
;------------------------------------------------------------------------------
emu_jabs:	emu_nextword			;get next PC word
		brtc	emu_jabs_1		;branch if no jump
		movw	PC_L,XL			;set PC
emu_jabs_1:	rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; call if T is set
;------------------------------------------------------------------------------
emu_rst:	ldi	XH,0x00			;HI ADDR =0
		rjmp	emu_call_0
emu_call:	emu_nextword			;get next PC word to r_adr
		brtc	emu_jabs_1		;branch if no call
emu_call_0:	movw	r_datl,PC_L		;recent address
		movw	PC_L,XL			;new address
		emu_SP_write
emu_call_1:	rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; ret if T is set
;------------------------------------------------------------------------------
emu_ret:	brtc	emu_ret_1		;branch if no jump
		emu_SP_read
		movw	PC_L,r_datl		;2 get PC from r_dath/r_datl
emu_ret_1:	rjmp	emu_loop		;2 goto loop


;------------------------------------------------------------------------------
; increment 16 bit register
;------------------------------------------------------------------------------
emu_inc16:	ld	XL,Y
		ldd	XH,Y+1
		adiw	XL,1
		st	Y,XL
		std	Y+1,XH
		rjmp	emu_nloop		;2 goto loop


;------------------------------------------------------------------------------
; decrement 16 bit register
;------------------------------------------------------------------------------
emu_dec16:	ld	XL,Y
		ldd	XH,Y+1
		sbiw	XL,1
		st	Y,XL
		std	Y+1,XH
		rjmp	emu_nloop		;2 goto loop


;------------------------------------------------------------------------------
; push word (Y)
;------------------------------------------------------------------------------
emu_pushword:	ld	r_datl,Y		;get register low
		ldd	r_dath,Y+1		;get register high
		emu_SP_write			;put register to stack
		rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; pop word (Y)
;------------------------------------------------------------------------------
emu_popword:	movw	XL,r0			;get SP
		emu_SP_read
		st	Y,r_datl
		std	Y+1,r_dath
		rjmp	emu_loop		;2 goto loop

;------------------------------------------------------------------------------
; the main emulator loop (neutral)
;------------------------------------------------------------------------------
emu_loop:	emu_pcaddr_read			;set addr in extmem to PC
emu_nloop:
;		lds	ZL,libmio_keycode	;debug
;		cpi	ZL,0xf8
;		brne	emu_nloop2
;		rjmp	debug_end
emu_nloop2:	emu_nextbyte			;get CMD byte
		ijmp				;branch


;------------------------------------------------------------------------------
; 0x01		LD BC,nnnn
;------------------------------------------------------------------------------
emu_code_01:	ldi	YL,12			;set ptr to BC
		rjmp	emu_setiw		;2 get next PC word and write to BC

;------------------------------------------------------------------------------
; 0x02		LD (BC),A
;------------------------------------------------------------------------------
emu_code_02:	ldi	YL,13			;set ptr to B
		rjmp	emu_astore		;2

;------------------------------------------------------------------------------
; 0x03		INC BC
;------------------------------------------------------------------------------
emu_code_03:	ldi	YL,12			;set ptr to BC
		rjmp	emu_inc16		;2 end of command

;------------------------------------------------------------------------------
; 0x04		INC B
;------------------------------------------------------------------------------
emu_code_04:	ldi	YL,13			;set ptr to B
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x05		DEC B
;------------------------------------------------------------------------------
emu_code_05:	ldi	YL,13			;set ptr to B
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x06		LD	B,nn
;------------------------------------------------------------------------------
emu_code_06:	ldi	YL,13			;set ptr to B
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x07		RLC	A
;------------------------------------------------------------------------------
emu_code_07:	bst	reg_a,7			;1
		bld	reg_f,0			;1
		lsl	reg_a			;1
		bld	reg_a,0			;1
		andi	reg_f,0xed		;1 clear N+H flags
		rjmp	emu_nloop		;2 end of command

;------------------------------------------------------------------------------
; 0x09		ADD	HL,BC
;------------------------------------------------------------------------------
emu_code_09:	ldi	YL,12			;set ptr to BC
		rjmp	emu_add16

;------------------------------------------------------------------------------
; 0x0a		LD	A,(BC)
;------------------------------------------------------------------------------
emu_code_0a:	movw	XL,reg_c
		rjmp	emu_areadx		;2 exec command

;------------------------------------------------------------------------------
; 0x0b		DEC	BC
;------------------------------------------------------------------------------
emu_code_0b:	ldi	YL,12			;set ptr to BC
		rjmp	emu_dec16		;2 exec command

;------------------------------------------------------------------------------
; 0x0c		INC C
;------------------------------------------------------------------------------
emu_code_0c:	ldi	YL,12			;set offset to reg C
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x0d		DEC 	C
;------------------------------------------------------------------------------
emu_code_0d:	ldi	YL,12			;set offset to reg C
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x0e		LD	C,nn
;------------------------------------------------------------------------------
emu_code_0e:	ldi	YL,12			;set offset to reg C
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x0f		RRC	A
;------------------------------------------------------------------------------
emu_code_0f:	bst	reg_a,0			;1
		bld	reg_f,0			;1
		lsr	reg_a			;1
		bld	reg_a,7			;1
		andi	reg_f,0xed		;1 clear N+H flags
		rjmp	emu_nloop		;2 end of command

;------------------------------------------------------------------------------
; 0x11		LD 	DE,nn
;------------------------------------------------------------------------------
emu_code_11:	ldi	YL,14			;set ptr to DE
		rjmp	emu_setiw		;get next PC word and write to BC

;------------------------------------------------------------------------------
; 0x12		LD 	(DE),A
;------------------------------------------------------------------------------
emu_code_12:	ldi	YL,14			;set ptr to DE
		rjmp	emu_astore		;2 end of command

;------------------------------------------------------------------------------
; 0x13		INC DE
;------------------------------------------------------------------------------
emu_code_13:	ldi	YL,14			;set ptr to DE
		rjmp	emu_inc16

;------------------------------------------------------------------------------
; 0x14		INC D
;------------------------------------------------------------------------------
emu_code_14:	ldi	YL,15			;set ptr to D
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x15		DEC D
;------------------------------------------------------------------------------
emu_code_15:	ldi	YL,15			;set ptr to D
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x16		LD	D,nn
;------------------------------------------------------------------------------
emu_code_16:	ldi	YL,15			;set ptr to D
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x17		RL	A
;------------------------------------------------------------------------------
emu_code_17:	mov	r_dath,reg_f		;1 get carry
		lsr	r_dath
		rol	reg_a
		rol	r_dath
		bld	r_dath,0
		bst	reg_f,0
		andi	reg_f,0xed		;1 clear N+H flags
		rjmp	emu_nloop		;2 end of command

;------------------------------------------------------------------------------
; 0x19		ADD	HL,DE
;------------------------------------------------------------------------------
emu_code_19:	ldi	YL,14			;set ptr to DE
		rjmp	emu_add16

;------------------------------------------------------------------------------
; 0x1a		LD	A,(DE)
;------------------------------------------------------------------------------
emu_code_1a:	movw	XL,reg_e
		rjmp	emu_areadx		;2 exec command

;------------------------------------------------------------------------------
; 0x1b		DEC	DE
;------------------------------------------------------------------------------
emu_code_1b:	ldi	YL,14			;set ptr to DE
		rjmp	emu_dec16

;------------------------------------------------------------------------------
; 0x1c		INC	E
;------------------------------------------------------------------------------
emu_code_1c:	ldi	YL,14			;set ptr to E
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x1d		DEC	E
;------------------------------------------------------------------------------
emu_code_1d:	ldi	YL,14			;set ptr to E
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x1e		LD	E,nn
;------------------------------------------------------------------------------
emu_code_1e:	ldi	YL,14			;set ptr to E
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x1f		RR	A
;------------------------------------------------------------------------------
emu_code_1f:	mov	r_dath,reg_f		;1 get carry
		lsr	r_dath
		ror	reg_a
		rol	r_dath
		bld	r_dath,0
		bst	reg_f,0
		andi	reg_f,0xed		;1 clear N+H flags
		rjmp	emu_nloop		;2 end of command


;------------------------------------------------------------------------------
; 0x21		LD 	HL,nnnn
;------------------------------------------------------------------------------
emu_code_21:	ldi	YL,22			;set ptr to H
		rjmp	emu_setiw		;get next PC word and write

;------------------------------------------------------------------------------
; 0x22		LD 	(nnnn),HL
;------------------------------------------------------------------------------
emu_code_22:	emu_nextword			;get addr to XH/XL
		movw	r_datl,reg_l		;copy data
		movw	XL,YL
		emu_wword			;write 2 bytes
		rjmp	emu_loop		;end of command

;------------------------------------------------------------------------------
; 0x23		INC	HL
;------------------------------------------------------------------------------
emu_code_23:	ldi	YL,22			;set ptr to HL
		rjmp	emu_inc16

;------------------------------------------------------------------------------
; 0x24		INC	H 
;------------------------------------------------------------------------------
emu_code_24:	ldi	YL,23			;set ptr to H
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x25		DEC	H
;------------------------------------------------------------------------------
emu_code_25:	ldi	YL,23			;set ptr to H
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x26		LD	H,nn
;------------------------------------------------------------------------------
emu_code_26:	ldi	YL,23			;set ptr to H
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x27		DA	A
;------------------------------------------------------------------------------
emu_code_27:	mov	r_dath,reg_a		;1 copy A
		andi	r_dath,0x0f		;1 mask bits
		cpi	r_dath,0x0a		;1 check if we must add
		sbrs	reg_f,4			;1 skip if H set
		brcs	emu_code_27a		;1/2
		subi	reg_a,0xfa		;1 +6
emu_code_27a:	mov	r_dath,reg_a		;1 copy A
		andi	r_dath,0xf0		;1 mask bits
		cpi	r_dath,0x0a		;1 check if we must add
		clr	XL			;1
		brcs	emu_code_27b		;1/2
		ldi	XL,0x60		;1 +60
emu_code_27b:	add	reg_a,XL
		brcc	emu_code_27c		;no carry
		ori	reg_f,1			;set carry
emu_code_27c:	mov	ZL,reg_a			;1
		ldi	ZH,HIGH(emu_flags_log*2)	;1
		lpm	ZL,Z			;1
		andi	reg_f,0x3b		;1
		or	reg_f,ZL			;1
		rjmp	emu_loop		;2




;------------------------------------------------------------------------------
; the level1 jump table
;------------------------------------------------------------------------------
.org 0x4a00

emu_jtab1:	rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_01
		rjmp	emu_code_02
		rjmp	emu_code_03
		rjmp	emu_code_04
		rjmp	emu_code_05
		rjmp	emu_code_06
		rjmp	emu_code_07
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_09
		rjmp	emu_code_0a
		rjmp	emu_code_0b
		rjmp	emu_code_0c
		rjmp	emu_code_0d
		rjmp	emu_code_0e
		rjmp	emu_code_0f

		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_11
		rjmp	emu_code_12
		rjmp	emu_code_13
		rjmp	emu_code_14
		rjmp	emu_code_15
		rjmp	emu_code_16
		rjmp	emu_code_17
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_19
		rjmp	emu_code_1a
		rjmp	emu_code_1b
		rjmp	emu_code_1c
		rjmp	emu_code_1d
		rjmp	emu_code_1e
		rjmp	emu_code_1f

		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_21
		rjmp	emu_code_22
		rjmp	emu_code_23
		rjmp	emu_code_24
		rjmp	emu_code_25
		rjmp	emu_code_26
		rjmp	emu_code_27
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_29
		rjmp	emu_code_2a
		rjmp	emu_code_2b
		rjmp	emu_code_2c
		rjmp	emu_code_2d
		rjmp	emu_code_2e
		rjmp	emu_code_2f

		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_31
		rjmp	emu_code_32
		rjmp	emu_code_33
		rjmp	emu_code_34
		rjmp	emu_code_35
		rjmp	emu_code_36
		rjmp	emu_code_37
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_39
		rjmp	emu_code_3a
		rjmp	emu_code_3b
		rjmp	emu_code_3c
		rjmp	emu_code_3d
		rjmp	emu_code_3e
		rjmp	emu_code_3f

		rjmp	emu_nloop		;2	LD B,B
		rjmp	emu_code_41
		rjmp	emu_code_42
		rjmp	emu_code_43
		rjmp	emu_code_44
		rjmp	emu_code_45
		rjmp	emu_code_46
		rjmp	emu_code_47
		rjmp	emu_code_48
		rjmp	emu_nloop		;2	LD C,C
		rjmp	emu_code_4a
		rjmp	emu_code_4b
		rjmp	emu_code_4c
		rjmp	emu_code_4d
		rjmp	emu_code_4e
		rjmp	emu_code_4f

		rjmp	emu_code_50
		rjmp	emu_code_51
		rjmp	emu_nloop		;2	LD D,D
		rjmp	emu_code_53
		rjmp	emu_code_54
		rjmp	emu_code_55
		rjmp	emu_code_56
		rjmp	emu_code_57
		rjmp	emu_code_58
		rjmp	emu_code_59
		rjmp	emu_code_5a
		rjmp	emu_nloop		;2	LD E,E
		rjmp	emu_code_5c
		rjmp	emu_code_5d
		rjmp	emu_code_5e
		rjmp	emu_code_5f

		rjmp	emu_code_60
		rjmp	emu_code_61
		rjmp	emu_code_62
		rjmp	emu_code_63
		rjmp	emu_nloop		;2	LD H,H
		rjmp	emu_code_65
		rjmp	emu_code_66
		rjmp	emu_code_67
		rjmp	emu_code_68
		rjmp	emu_code_69
		rjmp	emu_code_6a
		rjmp	emu_code_6b
		rjmp	emu_code_6c
		rjmp	emu_nloop		;2	LD L,L
		rjmp	emu_code_6e
		rjmp	emu_code_6f

		rjmp	emu_code_70
		rjmp	emu_code_71
		rjmp	emu_code_72
		rjmp	emu_code_73
		rjmp	emu_code_74
		rjmp	emu_code_75
		rjmp	emu_code_76
		rjmp	emu_code_77
		rjmp	emu_code_78
		rjmp	emu_code_79
		rjmp	emu_code_7a
		rjmp	emu_code_7b
		rjmp	emu_code_7c
		rjmp	emu_code_7d
		rjmp	emu_code_7e
		rjmp	emu_nloop		;2	LD A,A

		rjmp	emu_code_80
		rjmp	emu_code_81
		rjmp	emu_code_82
		rjmp	emu_code_83
		rjmp	emu_code_84
		rjmp	emu_code_85
		rjmp	emu_code_86
		rjmp	emu_code_87
		rjmp	emu_code_88
		rjmp	emu_code_89
		rjmp	emu_code_8a
		rjmp	emu_code_8b
		rjmp	emu_code_8c
		rjmp	emu_code_8d
		rjmp	emu_code_8e
		rjmp	emu_code_8f

		rjmp	emu_code_90
		rjmp	emu_code_91
		rjmp	emu_code_92
		rjmp	emu_code_93
		rjmp	emu_code_94
		rjmp	emu_code_95
		rjmp	emu_code_96
		rjmp	emu_code_97
		rjmp	emu_code_98
		rjmp	emu_code_99
		rjmp	emu_code_9a
		rjmp	emu_code_9b
		rjmp	emu_code_9c
		rjmp	emu_code_9d
		rjmp	emu_code_9e
		rjmp	emu_code_9f

		rjmp	emu_code_a0
		rjmp	emu_code_a1
		rjmp	emu_code_a2
		rjmp	emu_code_a3
		rjmp	emu_code_a4
		rjmp	emu_code_a5
		rjmp	emu_code_a6
		rjmp	emu_code_a7
		rjmp	emu_code_a8
		rjmp	emu_code_a9
		rjmp	emu_code_aa
		rjmp	emu_code_ab
		rjmp	emu_code_ac
		rjmp	emu_code_ad
		rjmp	emu_code_ae
		rjmp	emu_code_af

		rjmp	emu_code_b0
		rjmp	emu_code_b1
		rjmp	emu_code_b2
		rjmp	emu_code_b3
		rjmp	emu_code_b4
		rjmp	emu_code_b5
		rjmp	emu_code_b6
		rjmp	emu_code_b7
		rjmp	emu_code_b8
		rjmp	emu_code_b9
		rjmp	emu_code_ba
		rjmp	emu_code_bb
		rjmp	emu_code_bc
		rjmp	emu_code_bd
		rjmp	emu_code_be
		rjmp	emu_code_bf

		rjmp	emu_code_c0
		rjmp	emu_code_c1
		rjmp	emu_code_c2
		rjmp	emu_code_c3
		rjmp	emu_code_c4
		rjmp	emu_code_c5
		rjmp	emu_code_c6
		rjmp	emu_code_c7
		rjmp	emu_code_c8
		rjmp	emu_code_c9
		rjmp	emu_code_ca
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_cc
		rjmp	emu_code_cd
		rjmp	emu_code_ce
		rjmp	emu_code_cf

		rjmp	emu_code_d0
		rjmp	emu_code_d1
		rjmp	emu_code_d2
		rjmp	emu_code_d3
		rjmp	emu_code_d4
		rjmp	emu_code_d5
		rjmp	emu_code_d6
		rjmp	emu_code_d7
		rjmp	emu_code_d8
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_da
		rjmp	emu_code_db
		rjmp	emu_code_dc
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_de
		rjmp	emu_code_df

		rjmp	emu_code_e0
		rjmp	emu_code_e1
		rjmp	emu_code_e2
		rjmp	emu_code_e3
		rjmp	emu_code_e4
		rjmp	emu_code_e5
		rjmp	emu_code_e6
		rjmp	emu_code_e7
		rjmp	emu_code_e8
		rjmp	emu_code_e9
		rjmp	emu_code_ea
		rjmp	emu_code_eb
		rjmp	emu_code_ec
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_ee
		rjmp	emu_code_ef

		rjmp	emu_code_f0
		rjmp	emu_code_f1
		rjmp	emu_code_f2
		rjmp	emu_code_f3
		rjmp	emu_code_f4
		rjmp	emu_code_f5
		rjmp	emu_code_f6
		rjmp	emu_code_f7
		rjmp	emu_code_f8
		rjmp	emu_code_f9
		rjmp	emu_code_fa
		rjmp	emu_code_fb
		rjmp	emu_code_fc
		rjmp	emu_nloop		;2	0x00	NOP
		rjmp	emu_code_fe
		rjmp	emu_code_ff


;------------------------------------------------------------------------------
; 0x29		ADD	HL,HL
;------------------------------------------------------------------------------
emu_code_29:	ldi	YL,22			;set ptr to HL
		rjmp	emu_add16

;------------------------------------------------------------------------------
; 0x2a		LD 	HL,(nnnn)
;------------------------------------------------------------------------------
emu_code_2a:	emu_nextword
		emu_rword
		movw	reg_l,r_datl
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x2b		DEC	HL
;------------------------------------------------------------------------------
emu_code_2b:	ldi	YL,22			;set ptr to HL
		rjmp	emu_dec16

;------------------------------------------------------------------------------
; 0x2c		INC	L
;------------------------------------------------------------------------------
emu_code_2c:	ldi	YL,22			;set ptr to L
		rjmp	emu_incb

;------------------------------------------------------------------------------
; 0x2d		DEC	L
;------------------------------------------------------------------------------
emu_code_2d:	ldi	YL,22			;set ptr to L
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x2e		LD	L,nn
;------------------------------------------------------------------------------
emu_code_2e:	ldi	YL,22			;set ptr to L
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x2f		CPL
;------------------------------------------------------------------------------
emu_code_2f:	com	reg_a			;1 ones complement
		ori	reg_f,0x60		;1 set flags
		rjmp	emu_loop		;2 end of command

;------------------------------------------------------------------------------
; 0x31		LD 	SP,nnnn
;------------------------------------------------------------------------------
emu_code_31:	ldi	YL,0			;set ptr to SPL
		ldi	YH,1
		rjmp	emu_setiw		;2 end of command

;------------------------------------------------------------------------------
; 0x32		LD 	(nnnn),a
;------------------------------------------------------------------------------
emu_code_32:	emu_nextword			; get next PC word to X
		mov	r_datl,reg_a
		rjmp	emu_cstore

;------------------------------------------------------------------------------
; 0x33		INC	SP
;------------------------------------------------------------------------------
emu_code_33:	ldi	YL,0x4A			;set ptr to SPH
		rjmp	emu_inc16		;2 end of command

;------------------------------------------------------------------------------
; 0x34		INC	(HL)
;------------------------------------------------------------------------------
emu_code_34:	rcall	emu_gethl
		add	r_datl,const_1
		clt				;clear sub flag
		in	reg_f,SREG
		rjmp	emu_puthl

;------------------------------------------------------------------------------
; 0x35		DEC	(HL)
;------------------------------------------------------------------------------
emu_code_35:	rcall	emu_gethl
		sub	r_datl,const_1
		in	reg_f,SREG
		rjmp	emu_puthl

;------------------------------------------------------------------------------
; 0x36		LD 	(HL),nn
;------------------------------------------------------------------------------
emu_code_36:	emu_nextbyte
		rjmp	emu_puthl

;------------------------------------------------------------------------------
; 0x37		SCF
;------------------------------------------------------------------------------
emu_code_37:	andi	reg_f,0xed		;clear H,N
		ori	reg_f,0x01		;set carry
		rjmp	emu_loop		;goto loop



;------------------------------------------------------------------------------
; 0x39		ADD	HL,SP
;------------------------------------------------------------------------------
emu_code_39:	ldi	YL,0x4A			;set ptr to SPH
		rjmp	emu_add16

;------------------------------------------------------------------------------
; 0x3a		LD	A,(nnnn)
;------------------------------------------------------------------------------
emu_code_3a:	emu_nextword			; get next PC word
		rjmp	emu_areadx

;------------------------------------------------------------------------------
; 0x3b		DEC	SP
;------------------------------------------------------------------------------
emu_code_3b:	ldi	YL,0x4A			;set ptr to SPH
		rjmp	emu_dec16		;2 end of command

;------------------------------------------------------------------------------
; 0x3c		INC	A
;------------------------------------------------------------------------------
emu_code_3c:	ldi	YL,20			;reg_a
		rjmp	emu_incb



;------------------------------------------------------------------------------
; 0x3d		DEC	A
;------------------------------------------------------------------------------
emu_code_3d:	ldi	YL,20			;reg_a
		rjmp	emu_decb

;------------------------------------------------------------------------------
; 0x3e		LD	A,nn
;------------------------------------------------------------------------------
emu_code_3e:	ldi	YL,20			;reg_a
		rjmp	emu_setib

;------------------------------------------------------------------------------
; 0x3f		CCF
;------------------------------------------------------------------------------
emu_code_3f:	andi	reg_f,0x9e		;clear H,N,C
		rjmp	emu_nloop		;goto loop




;------------------------------------------------------------------------------
; 0x41		LD	B,C
;------------------------------------------------------------------------------
emu_code_41:	mov	reg_b,reg_c
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x42		LD	B,D
;------------------------------------------------------------------------------
emu_code_42:	mov	reg_b,reg_d
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x43		LD	B,E
;------------------------------------------------------------------------------
emu_code_43:	mov	reg_b,reg_e
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x44		LD	B,H
;------------------------------------------------------------------------------
emu_code_44:	mov	reg_b,reg_h
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x45		LD	B,L
;------------------------------------------------------------------------------
emu_code_45:	mov	reg_b,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x46		LD	B,(HL)
;------------------------------------------------------------------------------
emu_code_46:	rcall	emu_gethl		;get (HL)
		mov	reg_b,r_datl
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x47		LD	B,A
;------------------------------------------------------------------------------
emu_code_47:	mov	reg_b,reg_a
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x48		LD	C,B
;------------------------------------------------------------------------------
emu_code_48:	mov	reg_c,reg_b
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x4a		LD	C,D
;------------------------------------------------------------------------------
emu_code_4a:	mov	reg_c,reg_d
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x4b		LD	C,E
;------------------------------------------------------------------------------
emu_code_4b:	mov	reg_c,reg_e
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x4c		LD	C,H
;------------------------------------------------------------------------------
emu_code_4c:	mov	reg_c,reg_h
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x4d		LD	C,L
;------------------------------------------------------------------------------
emu_code_4d:	mov	reg_c,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x4e		LD	C,(HL)
;------------------------------------------------------------------------------
emu_code_4e:	rcall	emu_gethl		;get (HL)
		mov	reg_c,r_datl
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x4f		LD	C,A
;------------------------------------------------------------------------------
emu_code_4f:	mov	reg_c,reg_a
		rjmp	emu_nloop		;goto loop


;------------------------------------------------------------------------------
; 0x50		LD	D,B
;------------------------------------------------------------------------------
emu_code_50:	mov	reg_d,reg_b
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x51		LD	D,C
;------------------------------------------------------------------------------
emu_code_51:	mov	reg_d,reg_c
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x53		LD	D,E
;------------------------------------------------------------------------------
emu_code_53:	mov	reg_d,reg_e
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x54		LD	D,H
;------------------------------------------------------------------------------
emu_code_54:	mov	reg_d,reg_h
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x55		LD	D,L
;------------------------------------------------------------------------------
emu_code_55:	mov	reg_d,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x56		LD	D,(HL)
;------------------------------------------------------------------------------
emu_code_56:	rcall	emu_gethl		;get (HL)
		mov	reg_d,r_dath
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x57		LD	D,A
;------------------------------------------------------------------------------
emu_code_57:	mov	reg_d,reg_a
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x58		LD	E,B
;------------------------------------------------------------------------------
emu_code_58:	mov	reg_e,reg_b
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x59		LD	E,C
;------------------------------------------------------------------------------
emu_code_59:	mov	reg_e,reg_c
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x5a		LD	E,D
;------------------------------------------------------------------------------
emu_code_5a:	mov	reg_e,reg_d
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x5c		LD	E,H
;------------------------------------------------------------------------------
emu_code_5c:	mov	reg_e,reg_h
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x5d		LD	E,L
;------------------------------------------------------------------------------
emu_code_5d:	mov	reg_e,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x5e		LD	E,(HL)
;------------------------------------------------------------------------------
emu_code_5e:	rcall	emu_gethl		;get (HL)
		mov	reg_e,r_dath
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x5f		LD	E,A
;------------------------------------------------------------------------------
emu_code_5f:	mov	reg_e,reg_a
		rjmp	emu_nloop		;goto loop


;------------------------------------------------------------------------------
; 0x60		LD	H,B
;------------------------------------------------------------------------------
emu_code_60:	mov	reg_h,reg_b
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x61		LD	H,C
;------------------------------------------------------------------------------
emu_code_61:	mov	reg_h,reg_c
		rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0x62		LD	H,D
;------------------------------------------------------------------------------
emu_code_62:	mov	reg_h,reg_d
		rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0x63		LD	H,E
;------------------------------------------------------------------------------
emu_code_63:	mov	reg_h,reg_e
		rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0x65		LD	H,L
;------------------------------------------------------------------------------
emu_code_65:	mov	reg_h,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x66		LD	H,(HL)
;------------------------------------------------------------------------------
emu_code_66:	rcall	emu_gethl		;get (HL)
		mov	reg_h,r_datl
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x67		LD	H,A
;------------------------------------------------------------------------------
emu_code_67:	mov	reg_h,reg_a
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x68		LD	L,B
;------------------------------------------------------------------------------
emu_code_68:	mov	reg_l,reg_b
		rjmp	emu_nloop		;goto loop


;------------------------------------------------------------------------------
; 0x69		LD	L,C
;------------------------------------------------------------------------------
emu_code_69:	mov	reg_l,reg_c
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x6a		LD	L,D
;------------------------------------------------------------------------------
emu_code_6a:	mov	reg_l,reg_d
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x6b		LD	L,E
;------------------------------------------------------------------------------
emu_code_6b:	mov	reg_l,reg_e
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x6c		LD	L,H
;------------------------------------------------------------------------------
emu_code_6c:	mov	reg_l,reg_h
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x6e		LD	L,(HL)
;------------------------------------------------------------------------------
emu_code_6e:	rcall	emu_gethl		;get (HL)
		mov	reg_l,r_datl
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x6f		LD	L,A
;------------------------------------------------------------------------------
emu_code_6f:	mov	reg_l,reg_a
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x70		LD	(HL),B
;------------------------------------------------------------------------------
emu_code_70:	mov	r_datl,reg_b
		rjmp	emu_puthl

;------------------------------------------------------------------------------
; 0x71		LD	(HL),C
;------------------------------------------------------------------------------
emu_code_71:	mov	r_datl,reg_c
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x72		LD	(HL),D
;------------------------------------------------------------------------------
emu_code_72:	mov	r_datl,reg_d
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x73		LD	(HL),E
;------------------------------------------------------------------------------
emu_code_73:	mov	r_datl,reg_e
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x74		LD	(HL),H
;------------------------------------------------------------------------------
emu_code_74:	mov	r_datl,reg_h
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x75		LD	(HL),L
;------------------------------------------------------------------------------
emu_code_75:	mov	r_datl,reg_l
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x76		HALT
;------------------------------------------------------------------------------
emu_code_76:
debug_end:
		ret				;end here

;------------------------------------------------------------------------------
; 0x77		LD	(HL),A
;------------------------------------------------------------------------------
emu_code_77:	mov	r_datl,reg_a
		rjmp	emu_puthl		;store and goto loop

;------------------------------------------------------------------------------
; 0x78		LD	A,B
;------------------------------------------------------------------------------
emu_code_78:	mov	reg_a,reg_b
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x79		LD	A,C
;------------------------------------------------------------------------------
emu_code_79:	mov	reg_a,reg_c
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7a		LD	A,D
;------------------------------------------------------------------------------
emu_code_7a:	mov	reg_a,reg_d
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7b		LD	A,E
;------------------------------------------------------------------------------
emu_code_7b:	mov	reg_a,reg_e
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x7c		LD	A,H
;------------------------------------------------------------------------------
emu_code_7c:	mov	reg_a,reg_h
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x7d		LD	A,L
;------------------------------------------------------------------------------
emu_code_7d:	mov	reg_a,reg_l
		rjmp	emu_nloop		;goto loop

;------------------------------------------------------------------------------
; 0x7e		LD	A,(HL)
;------------------------------------------------------------------------------
emu_code_7e:	rcall	emu_gethl
		mov	reg_a,r_datl
		rjmp	emu_loop		;goto loop

;------------------------------------------------------------------------------
; 0x80		ADD	B
;------------------------------------------------------------------------------
emu_code_80:	mov	r_datl,reg_b
; add r_datl to accu
emu_add8:	add	reg_a,r_datl
emu_code_arin:	in	r_dath,SREG
		andi	r_dath,0x21			;H+C
		mov	ZL,reg_a			;1
		lsl	ZH
		subi	ZH,0xf9				;+7
		lpm	reg_f,Z				;3
		or	reg_f,r_dath			;1
		mov	ZH,jtab_hi			;1
		rjmp	emu_nloop			;2

;------------------------------------------------------------------------------
; 0x81		ADD	C
;------------------------------------------------------------------------------
emu_code_81:	mov	r_datl,reg_c
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x82		ADD	D
;------------------------------------------------------------------------------
emu_code_82:	mov	r_datl,reg_d
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x83		ADD	E
;------------------------------------------------------------------------------
emu_code_83:	mov	r_datl,reg_e
		rjmp	emu_add8		;set flags and goto loop


;------------------------------------------------------------------------------
; 0x84		ADD	H
;------------------------------------------------------------------------------
emu_code_84:	mov	r_datl,reg_h
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x85		ADD	L
;------------------------------------------------------------------------------
emu_code_85:	mov	r_datl,reg_l
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x86		ADD	(HL)
;------------------------------------------------------------------------------
emu_code_86:	rcall	emu_gethl
		add	reg_a,r_datl
emu_code_ari:	in	r_dath,SREG
		andi	r_dath,0x21			;H+C
		mov	ZL,reg_a			;1
		lsl	ZH
		subi	ZH,0xf9
		lpm	reg_f,Z				;3
		or	reg_f,r_dath			;1
		mov	ZH,jtab_hi			;1
		rjmp	emu_nloop			;2

;------------------------------------------------------------------------------
; 0x87		ADD	A
;------------------------------------------------------------------------------
emu_code_87:	mov	r_datl,reg_a
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x88		ADC	B
;------------------------------------------------------------------------------
emu_code_88:	mov	r_datl,reg_b
; add r_datl to accu with carry
emu_adc8:	clt
		sbrc	reg_f,0
		set
		adc	reg_a,r_datl
		rjmp	emu_code_arin		;2 goto loop

;------------------------------------------------------------------------------
; 0x89		ADC	C
;------------------------------------------------------------------------------
emu_code_89:	mov	r_datl,reg_c
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x8a		ADC	D
;------------------------------------------------------------------------------
emu_code_8a:	mov	r_datl,reg_d
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x8b		ADC	E
;------------------------------------------------------------------------------
emu_code_8b:	mov	r_datl,reg_e
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x8c		ADC	H
;------------------------------------------------------------------------------
emu_code_8c:	mov	r_datl,reg_h
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x8d		ADC	L
;------------------------------------------------------------------------------
emu_code_8d:	mov	r_datl,reg_l
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x8e		ADC	(HL)
;------------------------------------------------------------------------------
emu_code_8e:	rcall	emu_gethl
		clt
		sbrc	reg_f,0
		set		
		adc	reg_a,r_datl		
		rjmp	emu_code_ari		;2 goto loop

;------------------------------------------------------------------------------
; 0x8f		ADC	A
;------------------------------------------------------------------------------
emu_code_8f:	mov	r_datl,reg_a
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x90		SUB	B
;------------------------------------------------------------------------------
emu_code_90:	mov	r_datl,reg_b
; sub r_datl from accu
emu_sub8:	sub	reg_a,r_datl
		rjmp	emu_code_arin		;2 goto loop

;------------------------------------------------------------------------------
; 0x91		SUB	C
;------------------------------------------------------------------------------
emu_code_91:	mov	r_datl,reg_c
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x92		SUB	D
;------------------------------------------------------------------------------
emu_code_92:	mov	r_datl,reg_d
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x93		SUB	E
;------------------------------------------------------------------------------
emu_code_93:	mov	r_datl,reg_e
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x94		SUB	H
;------------------------------------------------------------------------------
emu_code_94:	mov	r_datl,reg_h
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x95		SUB	L
;------------------------------------------------------------------------------
emu_code_95:	mov	r_datl,reg_l
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x96		SUB	(HL)
;------------------------------------------------------------------------------
emu_code_96:	rcall	emu_gethl
		sub	reg_a,r_datl
		rjmp	emu_code_ari		;2 goto loop

;------------------------------------------------------------------------------
; 0x97		SUB	A
;------------------------------------------------------------------------------
emu_code_97:	mov	r_datl,reg_a
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x98		SBC	B
;------------------------------------------------------------------------------
emu_code_98:	mov	r_datl,reg_b
; sub r_datl from accu with carry
emu_sbc8:	clt
		sbrc	reg_f,0
		set		
		sbc	reg_a,r_datl		
		rjmp	emu_code_arin		;2 goto loop

;------------------------------------------------------------------------------
; 0x99		SBC	C
;------------------------------------------------------------------------------
emu_code_99:	mov	r_datl,reg_c
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x9a		SBC	D
;------------------------------------------------------------------------------
emu_code_9a:	mov	r_datl,reg_d
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x9b		SBC	E
;------------------------------------------------------------------------------
emu_code_9b:	mov	r_datl,reg_e
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x9c		SBC	H
;------------------------------------------------------------------------------
emu_code_9c:	mov	r_datl,reg_h
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x9d		SBC	L
;------------------------------------------------------------------------------
emu_code_9d:	mov	r_datl,reg_l
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0x9e		SBC	(HL)
;------------------------------------------------------------------------------
emu_code_9e:	rcall	emu_gethl
		clt
		sbrc	reg_f,0
		set		
		sbc	reg_a,r_datl		
		rjmp	emu_code_ari		;2 goto loop

;------------------------------------------------------------------------------
; 0x9f		SBC	A
;------------------------------------------------------------------------------
emu_code_9f:	mov	r_datl,reg_a
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xa0		AND	B
;------------------------------------------------------------------------------
emu_code_a0:	mov	r_datl,reg_b
emu_code_and:	and	reg_a,r_datl
emu_code_logn:	mov	ZL,reg_a
		lsl	ZH
		subi	ZH,0xf9				;+7
		lpm	reg_f,Z				;3
		mov	ZH,jtab_hi			;1
		rjmp	emu_nloop			;2

;------------------------------------------------------------------------------
; 0xa1		AND	C
;------------------------------------------------------------------------------
emu_code_a1:	mov	r_datl,reg_c			;get C
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa2		AND	D
;------------------------------------------------------------------------------
emu_code_a2:	mov	r_datl,reg_d			;get D
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa3		AND	E
;------------------------------------------------------------------------------
emu_code_a3:	mov	r_datl,reg_e			;get E
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa4		AND	H
;------------------------------------------------------------------------------
emu_code_a4:	mov	r_datl,reg_h
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa5		AND	L
;------------------------------------------------------------------------------
emu_code_a5:	mov	r_datl,reg_l
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa6		AND	(HL)
;------------------------------------------------------------------------------
emu_code_a6:	rcall	emu_gethl
		and	reg_a,r_datl
emu_code_log:	mov	ZL,reg_a
		lsl	ZH
		subi	ZH,0xf9
		lpm	reg_f,Z				;3
		mov	ZH,jtab_hi			;1
		rjmp	emu_loop			;2

;------------------------------------------------------------------------------
; 0xa7		AND	A
;------------------------------------------------------------------------------
emu_code_a7:	mov	r_datl,reg_a			;get A
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xa8		XOR	B
;------------------------------------------------------------------------------
emu_code_a8:	mov	r_datl,reg_b			;get B
emu_code_xor:	eor	reg_a,r_datl
		rjmp	emu_code_logn			;2

;------------------------------------------------------------------------------
; 0xa9		XOR	C
;------------------------------------------------------------------------------
emu_code_a9:	mov	r_datl,reg_c			;get C
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xaa		XOR	D
;------------------------------------------------------------------------------
emu_code_aa:	mov	r_datl,reg_d			;get D
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xab		XOR	E
;------------------------------------------------------------------------------
emu_code_ab:	mov	r_datl,reg_e			;get E
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xac		XOR	H
;------------------------------------------------------------------------------
emu_code_ac:	mov	r_datl,reg_h
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xad		XOR	L
;------------------------------------------------------------------------------
emu_code_ad:	mov	r_datl,reg_l
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xae		XOR	(HL)
;------------------------------------------------------------------------------
emu_code_ae:	rcall	emu_gethl
		eor	reg_a,r_datl
		rjmp	emu_code_log

;------------------------------------------------------------------------------
; 0xaf		XOR	A
;------------------------------------------------------------------------------
emu_code_af:	mov	r_datl,reg_a			;get A
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xb0		OR	B
;------------------------------------------------------------------------------
emu_code_b0:	mov	r_datl,reg_b			;get B
emu_code_or:	or	reg_a,r_datl
		rjmp	emu_code_logn			;2

;------------------------------------------------------------------------------
; 0xb1		OR	C
;------------------------------------------------------------------------------
emu_code_b1:	mov	r_datl,reg_c			;get C
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb2		OR	D
;------------------------------------------------------------------------------
emu_code_b2:	mov	r_datl,reg_d			;get D
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb3		OR	E
;------------------------------------------------------------------------------
emu_code_b3:	mov	r_datl,reg_e			;get E
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb4		OR	H
;------------------------------------------------------------------------------
emu_code_b4:	mov	r_datl,reg_h
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb5		OR	L
;------------------------------------------------------------------------------
emu_code_b5:	mov	r_datl,reg_l
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb6		OR	(HL)
;------------------------------------------------------------------------------
emu_code_b6:	rcall	emu_gethl
		or	reg_a,r_datl
		rjmp	emu_code_log

;------------------------------------------------------------------------------
; 0xb7		OR	A
;------------------------------------------------------------------------------
emu_code_b7:	mov	r_datl,reg_a			;get A
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xb8		CMP	B
;------------------------------------------------------------------------------
emu_code_b8:	mov	r_datl,reg_b
; compare r_datl with accu
emu_cmp8:	cp	reg_a,r_datl
		rjmp	emu_code_arin		;2 goto loop

;------------------------------------------------------------------------------
; 0xb9		CMP	C
;------------------------------------------------------------------------------
emu_code_b9:	mov	r_datl,reg_c			;get C
		rjmp	emu_cmp8

;------------------------------------------------------------------------------
; 0xba		CMP	D
;------------------------------------------------------------------------------
emu_code_ba:	mov	r_datl,reg_d			;get D
		rjmp	emu_cmp8

;------------------------------------------------------------------------------
; 0xbb		CMP	E
;------------------------------------------------------------------------------
emu_code_bb:	mov	r_datl,reg_e			;get E
		rjmp	emu_cmp8

;------------------------------------------------------------------------------
; 0xbc		CMP	H
;------------------------------------------------------------------------------
emu_code_bc:	mov	r_datl,reg_h
		rjmp	emu_cmp8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xbd		CMP	L
;------------------------------------------------------------------------------
emu_code_bd:	mov	r_datl,reg_l
		rjmp	emu_cmp8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xbe		CMP	(HL)
;------------------------------------------------------------------------------
emu_code_be:	rcall	emu_gethl
		cp	reg_a,r_datl
		rjmp	emu_code_arin		;2 goto loop

;------------------------------------------------------------------------------
; 0xbf		CMP	A
;------------------------------------------------------------------------------
emu_code_bf:	mov	r_dath,reg_a		;get A
		rjmp	emu_cmp8

;------------------------------------------------------------------------------
; 0xc0		RNZ
;------------------------------------------------------------------------------
emu_code_c0:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,1			;/Z
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xc1		POP	BC
;------------------------------------------------------------------------------
emu_code_c1:	ldi	YL,12
		rjmp	emu_popword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xc2		JPNZ	nnnn
;------------------------------------------------------------------------------
emu_code_c2:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,1		;/Z
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xc3		JMP	nnnn
;------------------------------------------------------------------------------
emu_code_c3:	set
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xc4		CANZ	nnnn
;------------------------------------------------------------------------------
emu_code_c4:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,1			;/Z
		rjmp	emu_call		;goto loop

;------------------------------------------------------------------------------
; 0xc5		PUSH	BC
;------------------------------------------------------------------------------
emu_code_c5:	ldi	YL,12
		rjmp	emu_pushword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xc6		ADD	nn
;------------------------------------------------------------------------------
emu_code_c6:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_add8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xc7		RST	00H
;------------------------------------------------------------------------------
emu_code_c7:	ldi	XL,0x00
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xc8		RZ
;------------------------------------------------------------------------------
emu_code_c8:	bst	reg_f,1
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xc9		RET
;------------------------------------------------------------------------------
emu_code_c9:	set
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xca		JPZ	nnnn
;------------------------------------------------------------------------------
emu_code_ca:	bst	reg_f,1
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xcc		CAZ	nnnn
;------------------------------------------------------------------------------
emu_code_cc:	bst	reg_f,1
		rjmp	emu_call		;goto loop

;------------------------------------------------------------------------------
; 0xcd		CALL	nnnn
;------------------------------------------------------------------------------
emu_code_cd:	set
		rjmp	emu_call		;goto loop

;------------------------------------------------------------------------------
; 0xce		ADC	nn
;------------------------------------------------------------------------------
emu_code_ce:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_adc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xcf		RST	08H
;------------------------------------------------------------------------------
emu_code_cf:	ldi	XL,0x08
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xd0		RNC
;------------------------------------------------------------------------------
emu_code_d0:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,0			;/C
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xd1		POP	DE
;------------------------------------------------------------------------------
emu_code_d1:	ldi	YL,14
		rjmp	emu_popword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xd2		JPNC	nnnn
;------------------------------------------------------------------------------
emu_code_d2:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,0			;/C
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xd3		out	nn
;------------------------------------------------------------------------------
emu_code_d3:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_output		;goto loop

;------------------------------------------------------------------------------
; 0xd4		CANC	nnnn
;------------------------------------------------------------------------------
emu_code_d4:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,0			;/C
		rjmp	emu_call		;goto loop

;------------------------------------------------------------------------------
; 0xd5		PUSH	DE
;------------------------------------------------------------------------------
emu_code_d5:	ldi	YL,14
		rjmp	emu_pushword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xd6		SUB	nn
;------------------------------------------------------------------------------
emu_code_d6:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_sub8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xd7		RST	10H
;------------------------------------------------------------------------------
emu_code_d7:	ldi	XL,0x10
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xd8		RC
;------------------------------------------------------------------------------
emu_code_d8:	bst	reg_f,0
		rjmp	emu_ret			;goto loop


;------------------------------------------------------------------------------
; 0xda		JPC	nnnn
;------------------------------------------------------------------------------
emu_code_da:	bst	reg_f,0
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xdb		IN	nn
;------------------------------------------------------------------------------
emu_code_db:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_input		;goto loop

;------------------------------------------------------------------------------
; 0xdc		CAC	nnnn
;------------------------------------------------------------------------------
emu_code_dc:	bst	reg_f,0
		rjmp	emu_call		;goto loop


;------------------------------------------------------------------------------
; 0xde		SBC	nn
;------------------------------------------------------------------------------
emu_code_de:	emu_nextbyte			;8 get next PC byte
		rjmp	emu_sbc8		;set flags and goto loop

;------------------------------------------------------------------------------
; 0xdf		RST	18H
;------------------------------------------------------------------------------
emu_code_df:	ldi	XL,0x18
		rjmp	emu_rst			;goto loop


;------------------------------------------------------------------------------
; 0xe0		RPO
;------------------------------------------------------------------------------
emu_code_e0:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,3			;/P
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xe1		POP	HL
;------------------------------------------------------------------------------
emu_code_e1:	ldi	YL,22
		rjmp	emu_popword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xe2		JPPO nnnn
;------------------------------------------------------------------------------
emu_code_e2:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,3			;/P
		rjmp	emu_jabs			;goto loop


;------------------------------------------------------------------------------
; 0xe3		EX (SP),HL
;------------------------------------------------------------------------------
emu_code_e3:	movw	XL,r0			;
		emu_rword
		adiw	XL,2
		movw	r0,XL
		movw	YL,r_datl
		movw	r_datl,reg_l
		movw	reg_l,YL
		ldi	YL,16
		clr	YH
		rjmp	emu_pushword

;------------------------------------------------------------------------------
; 0xe4		CAPO nnnn
;------------------------------------------------------------------------------
emu_code_e4:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,3			;/P
		rjmp	emu_call			;goto loop


;------------------------------------------------------------------------------
; 0xe5		PUSH HL
;------------------------------------------------------------------------------
emu_code_e5:	ldi	YL,22
		rjmp	emu_pushword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xe6		AND nn
;------------------------------------------------------------------------------
emu_code_e6:	emu_nextbyte
		rjmp	emu_code_and

;------------------------------------------------------------------------------
; 0xe7		RST	20H
;------------------------------------------------------------------------------
emu_code_e7:	ldi	XL,0x20
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xe8		RPE
;------------------------------------------------------------------------------
emu_code_e8:	bst	reg_f,3			;P
		rjmp	emu_ret			;goto loop


;------------------------------------------------------------------------------
; 0xe9		JMP HL
;------------------------------------------------------------------------------
emu_code_e9:	movw	PC_L,reg_l
		rjmp	emu_loop

;------------------------------------------------------------------------------
; 0xea		JPPE nnnn
;------------------------------------------------------------------------------
emu_code_ea:	bst	reg_f,3			;P
		rjmp	emu_jabs			;goto loop

;------------------------------------------------------------------------------
; 0xeb		EX 	DE,HL
;------------------------------------------------------------------------------
emu_code_eb:	movw	XL,reg_e
		movw	reg_e,reg_l
		movw	reg_l,XL
		rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0xec		CAPE nnnn
;------------------------------------------------------------------------------
emu_code_ec:	bst	reg_f,3			;P
		rjmp	emu_call			;goto loop


;------------------------------------------------------------------------------
; 0xee		XOR n
;------------------------------------------------------------------------------
emu_code_ee:	emu_nextbyte
		rjmp	emu_code_xor

;------------------------------------------------------------------------------
; 0xef		RST	28H
;------------------------------------------------------------------------------
emu_code_ef:	ldi	XL,0x28
		rjmp	emu_rst			;goto loop

;------------------------------------------------------------------------------
; 0xf0		RP
;------------------------------------------------------------------------------
emu_code_f0:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,2			;/S
		rjmp	emu_ret			;goto loop

;------------------------------------------------------------------------------
; 0xf1		POP AF
;------------------------------------------------------------------------------
emu_code_f1:	ldi	YL,20
		rjmp	emu_popword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xf2		JPP nnnn
;------------------------------------------------------------------------------
emu_code_f2:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,2			;/S
		rjmp	emu_jabs			;goto loop

;------------------------------------------------------------------------------
; 0xf3		DI
;------------------------------------------------------------------------------
emu_code_f3:	rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0xf4		CAP nnnn
;------------------------------------------------------------------------------
emu_code_f4:	mov	r_dath,reg_f
		com	r_dath
		bst	r_dath,2			;/S
		rjmp	emu_call			;goto loop

;------------------------------------------------------------------------------
; 0xf5		PUSH AF
;------------------------------------------------------------------------------
emu_code_f5:	ldi	YL,20
		rjmp	emu_pushword		;get two bytes from stack

;------------------------------------------------------------------------------
; 0xf6		OR n
;------------------------------------------------------------------------------
emu_code_f6:	emu_nextbyte
		rjmp	emu_code_or

;------------------------------------------------------------------------------
; 0xf7		RST	30H
;------------------------------------------------------------------------------
emu_code_f7:	ldi	XL,0x30
		rjmp	emu_rst			;goto loop


;------------------------------------------------------------------------------
; 0xf8		RM
;------------------------------------------------------------------------------
emu_code_f8:	bst	reg_f,2			;S
		rjmp	emu_ret			;goto loop


;------------------------------------------------------------------------------
; 0xf9		LD	SP,HL
;------------------------------------------------------------------------------
emu_code_f9:	ldi	YL,0x4A
		st	Y,reg_l
		std	Y+1,reg_h
		rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0xfa		JPM nnnn
;------------------------------------------------------------------------------
emu_code_fa:	bst	reg_f,3			;P
		rjmp	emu_jabs		;goto loop

;------------------------------------------------------------------------------
; 0xfb		EI
;------------------------------------------------------------------------------
emu_code_fb:	rjmp	emu_nloop

;------------------------------------------------------------------------------
; 0xfc		CAM nnnn
;------------------------------------------------------------------------------
emu_code_fc:	bst	reg_f,3			;P
		rjmp	emu_call		;goto loop


;------------------------------------------------------------------------------
; 0xfe		CMP nn
;------------------------------------------------------------------------------
emu_code_fe:	emu_nextbyte
		rjmp	emu_cmp8

;------------------------------------------------------------------------------
; 0xff		RST	38H
;------------------------------------------------------------------------------
emu_code_ff:	ldi	XL,0x38
		rjmp	emu_rst			;goto loop


;The hw is modelled to make writing a CPM BIOS easier.
;Ports:
;IN(x)
;0 - Con status. Returns 0xFF if the UART has a byte, 0 otherwise.
;1 - Console input, aka UDR.
;OUT(X)
;2 - Console output
;16 - Track select
;18 - Sector select
;20 - Write addr l
;21 - Write addr h
;22 - Trigger - write 1 to read, 2 to write a sector using the above info.
;	This will automatically move track, sector and dma addr to the next sector.


emu_input:	cpi	ZL,0x00			;console status
		brne	emu_input_10
		lds	reg_a,libmio_keycode
		cpi	reg_a,0
		breq	emu_input_1
		ldi	reg_a,0xff
emu_input_1:	rjmp	emu_nloop

emu_input_10:	lds	reg_a,libmio_lastkey
		rjmp	emu_nloop

emu_output:	cpi	ZL,0x02
		brne	emu_output_10
emu_charout:	push	reg_a
		api_outchar
		pop	reg_a
		rjmp	emu_nloop
		
emu_output_10:	mov	XL,reg_a
		api_setborder
		rjmp	emu_nloop


.org 0x4d80
emu_flags_log:

	.db	0x0a,0x00,0x00,0x08,	0x00,0x08,0x08,0x00,	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08 ;0n
	.db	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08,	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00 ;1n
	.db	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08,	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00 ;2n
	.db	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00,	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08 ;3n

	.db	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08,	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00 ;4n
	.db	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00,	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08 ;5n
	.db	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00,	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08 ;6n
	.db	0x00,0x08,0x08,0x00,	0x08,0x00,0x00,0x08,	0x08,0x00,0x00,0x08,	0x00,0x08,0x08,0x00 ;7n

	.db	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c,	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04 ;8n
	.db	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04,	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c ;9n
	.db	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04,	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c ;an
	.db	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c,	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04 ;bn

	.db	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04,	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c ;cn
	.db	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c,	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04 ;dn
	.db	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c,	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04 ;en
	.db	0x0c,0x04,0x04,0x0c,	0x04,0x0c,0x0c,0x04,	0x04,0x0c,0x0c,0x04,	0x0c,0x04,0x04,0x0c ;fn

emu_flags_a:
